import type { Router } from 'vue-router';
import { defineStore } from 'pinia';
import { EnumStorageKey } from '@/enum';
import { routes as staticRoutes } from '@/router';
import { fetchUserRoutes, fetchUserInfo } from '@/service';
import {
  getUserInfo,
  transformAuthRouteToMenu,
  transformAuthRoutesToVueRoutes,
  transformAuthRoutesToSearchMenus,
  getCacheRoutes,
  setLocal,
  getLocal
} from '@/utils';
import { useTabStore } from '../tab';
/** 扩展路由 */
import extendRoutes from './extendRoutes';

/** 将路由权限插入到第二级 */
function insertToLevel2(sourceData: any, targetData: any, key: string) {
  const target = targetData.find((r: { name: string }) => r.name === key);
  if (!target) return;
  console.log(target);
  sourceData.forEach((r: any) => {
    target.children.push(r);
  });
}

interface RouteState {
  /**
   * 权限路由模式:
   * - static - 前端声明的静态
   * - dynamic - 后端返回的动态
   */
  authRouteMode: ImportMetaEnv['VITE_AUTH_ROUTE_MODE'];
  /** 是否初始化了权限路由 */
  isInitedAuthRoute: boolean;
  /** 路由首页name(前端静态路由时生效，后端动态路由该值会被后端返回的值覆盖) */
  routeHomeName: AuthRoute.RouteKey;
  /** 菜单 */
  menus: GlobalMenuOption[];
  /** 搜索的菜单 */
  searchMenus: AuthRoute.Route[];
  /** 缓存的路由名称 */
  cacheRoutes: string[];
}

export const useRouteStore = defineStore('route-store', {
  state: (): RouteState => ({
    authRouteMode: import.meta.env.VITE_AUTH_ROUTE_MODE,
    isInitedAuthRoute: false,
    routeHomeName: 'home',
    menus: [],
    searchMenus: [],
    cacheRoutes: []
  }),
  actions: {
    /**
     * 处理权限路由
     * @param routes - 权限路由
     * @param router - 路由实例
     */
    handleAuthRoutes(routes: AuthRoute.Route[], router: Router) {
      this.menus = transformAuthRouteToMenu(routes);
      this.searchMenus = transformAuthRoutesToSearchMenus(routes);

      const vueRoutes = transformAuthRoutesToVueRoutes(routes);
      vueRoutes.forEach(route => {
        router.addRoute(route);
      });

      this.cacheRoutes = getCacheRoutes(vueRoutes);
      console.log(this.menus);
    },
    /**
     * 初始化动态路由
     * @param router - 路由实例
     */
    async initDynamicRoute(router: Router) {
      const { data } = await fetchUserRoutes();
      const routerData = data as any;
      if (routerData && routerData?.data.length > 0) {
        this.routeHomeName = 'home';
        /** 这里需要初始化本地的 三级页面。例如新增/编辑页面 */
        // 临时处理不去首页
        routerData.data.forEach((item: any) => {
          if (!item.children && item.meta?.singleLayout !== 'basic') {
            const citem = item;
            citem.meta.singleLayout = 'basic';
          }
        });
        extendRoutes.baseRoutes[0].redirect = routerData.data[0]?.path;
        this.handleAuthRoutes([...extendRoutes.baseRoutes, ...routerData.data], router);
      }
      router.replace({
        name: 'no-permission',
        path: 'no-permission'
      });
    },
    /**
     * 初始化静态路由
     * @param router - 路由实例
     */
    async initStaticRoute(router: Router) {
      // 先根据用户权限过滤一下staticRoutes

      this.handleAuthRoutes(staticRoutes, router);
    },
    /**
     * 初始化权限路由
     * @param router - 路由实例
     */
    async initAuthRoute(router: Router) {
      const { initHomeTab } = useTabStore();
      const { userId } = getUserInfo();
      if (!userId) return;

      const isDynamicRoute = this.authRouteMode === 'dynamic';
      if (isDynamicRoute) {
        await this.initDynamicRoute(router);
      } else {
        await this.initStaticRoute(router);
      }

      initHomeTab(this.routeHomeName, router);
      this.isInitedAuthRoute = true;
    },
    /** 初始化用户信息 */
    async initUserInfo() {
      const { data } = await fetchUserInfo();
      if (data) {
        setLocal(EnumStorageKey.permissions, data.data.permissions);
      }
    }
  }
});
