import { unref } from 'vue';
import { defineStore } from 'pinia';
import { useRouterPush } from '@/composables';
import { fetchLogin, fetchUserInfo } from '@/service';
import { getUserInfo, getToken, setUserInfo, setToken, clearAuthStorage } from '@/utils';

interface AuthState {
  /** 用户信息 */
  userInfo: Auth.UserInfo;
  /** 用户token */
  token: string;
  /** 登录的加载状态 */
  loginLoding: boolean;
  /** 权限合集 */
  permissions: any;
}

export const useAuthStore = defineStore('auth-store', {
  state: (): AuthState => ({
    userInfo: getUserInfo(),
    token: getToken(),
    loginLoding: false,
    permissions: []
  }),
  getters: {
    /** 是否登录 */
    isLogin(state) {
      return Boolean(state.token);
    }
  },
  actions: {
    /** 重置auth状态 */
    resetAuthStore() {
      const { toLogin } = useRouterPush(false);
      clearAuthStorage();
      this.$reset();
      toLogin();
      window.location.reload();
    },
    /**
     * 根据token进行登录
     * @param backendToken - 返回的token
     */
    async loginByToken(backendToken: ApiAuth.Token) {
      const { toHome } = useRouterPush(false);
      // 先把token存储到缓存中
      const { token } = backendToken;
      setToken(token);
      // 获取用户信息
      const { data } = await fetchUserInfo();
      if (data) {
        const userInfoObj = data.data.user;
        // 成功后把用户信息存储到缓存中
        setUserInfo(userInfoObj);

        // 更新状态
        Object.assign(this, { userInfo: userInfoObj, token });
        // 跳转登录后的地址
        toHome();

        // 登录成功弹出欢迎提示
        window.$notification?.success({
          title: '登录成功!',
          content: `欢迎回来，${userInfoObj.nickName}!`,
          closable: false,
          duration: 2500
        });

        return;
      }
      // 不成功则重置状态
      this.resetAuthStore();
    },
    /**
     * 登录
     * @param userName - 手机号
     * @param pwd - 密码或验证码
     * @param uuid - login code码的uuid
     * @param code - login code
     */
    async login(userName: string, pwd: string, uuid: string, code: string, errorCallback?: any) {
      this.loginLoding = true;
      const { data, error } = await fetchLogin(userName, pwd, uuid, code);
      if (data) {
        await this.loginByToken(data.data);
      } else if (typeof errorCallback === 'function') errorCallback(error);
      this.loginLoding = false;
    },

    /** 设置用户权限合集 */
    setPermissions(permissions: any) {
      this.permissions = [...permissions];
    }
  }
});
