import React, { useEffect } from 'react'
import { useLocation, useNavigate } from 'react-router-dom'
import { clearCurrentConversation, setCurrentConversation } from '@/store/conversationSlice'
import { clearCurrentTacticsConversation, setCurrentTacticsConversation } from '@/store/tacticsSlice'
import { useAppDispatch, useAppSelector } from '@/store/hook'
import { setIsAsking } from '@/store/chatSlice'
import type { RootState } from '@/store'
import { safeLocalStorageGetItem } from '@/lib/utils'

export function withRouteChangeHandler(WrappedComponent: React.ComponentType) {
  let beforeLocationPathName = ''
  return (props: any) => {
    const isAsking = useAppSelector((state: RootState) => state.chat.isAsking)
    const location = useLocation()
    const dispatch = useAppDispatch()
    const navigate = useNavigate()

    const newPath = location.pathname
    const oldPath = beforeLocationPathName

    useEffect(() => {
      // 处理 /home?from=tactics 重定向到 /tactics
      if (location.pathname === '/home') {
        const searchParams = new URLSearchParams(location.search)
        const from = searchParams.get('from')
        if (from === 'tactics') {
          // 重定向到 /tactics，保留所有查询参数（不再使用 sessionStorage）
          const newSearch = location.search
          navigate(`/tactics${newSearch}`, { replace: true })
          return
        }
      }

      if (isAsking && newPath !== oldPath && oldPath !== '') {
        dispatch(setIsAsking(false))
      }

      if (location.pathname === '/') {
        // 如果 localStorage 中有 conversationId，不清除，让恢复逻辑处理
        // 这样可以保留上一次的历史记录
        const savedConversationId = safeLocalStorageGetItem('currentConversationId')
        if (!savedConversationId) {
          dispatch(clearCurrentConversation())
        }
      }
      else if (location.pathname.startsWith('/chat/')) {
        const conversationId = location.pathname.split('/')[2]

        const tokenStr = window.localStorage.getItem('__TOKEN__') || '""'
        let token = ''
        try {
          token = JSON.parse(tokenStr)
        }
        catch {
          // 如果解析失败，说明没有有效的 token
          navigate('/')
          return // 提前返回，不执行后续的 dispatch
        }

        if (!token) {
          // 如果没有有效的 token，重定向到首页
          navigate('/')
          return // 提前返回，不执行后续的 dispatch
        }

        dispatch(setCurrentConversation(conversationId))
      }
      // 处理问答功能的聊天路由
      else if (location.pathname.startsWith('/tactics/chat/')) {
        const conversationId = location.pathname.split('/')[3]

        const tokenStr = window.localStorage.getItem('__TOKEN__') || '""'
        let token = ''
        try {
          token = JSON.parse(tokenStr)
        }
        catch {
          navigate('/tactics')
          return
        }

        if (!token) {
          navigate('/tactics')
          return
        }

        dispatch(setCurrentTacticsConversation(conversationId))
      }
      else if (location.pathname === '/tactics') {
        dispatch(clearCurrentTacticsConversation())
      }
      // 这里可以添加其他路由相关的逻辑
      beforeLocationPathName = newPath
    }, [location, dispatch, navigate])

    return <WrappedComponent {...props} />
  }
}
