import React, { useCallback, useEffect, useRef, useState } from 'react'
import { useLocation, useParams, useSearchParams } from 'react-router-dom'
import { Button } from '@heroui/react'
import { motion } from 'framer-motion'
import { useScroll } from 'ahooks'
import styles from './Chat.module.less'
import { processApiResponse } from './helper'
import { ChatWelcome } from './components/ChatWelcome'
import { ChatItemUser } from './components/ChatItem/ChatItemUser'
import { ChatAnswerBox } from './components/ChatItem/ChatAnswerBox'
import { trimSpacesOnly } from './components/ChatItem/markdownFormatter'
import { ChatEditor } from '@/components/ChatEditor'
import type { ChatRecord } from '@/types/chat'
import { fetchSessionConversationId, fetchUserQaRecordPage } from '@/api/conversation'
import { fetchCheckTokenApi, fetchStreamResponse } from '@/api/chat'
import { fetchToolList } from '@/api/home'
import { clearCurrentToolId, clearShouldSendQuestion, fetchConversations, setCurrentToolId } from '@/store/conversationSlice'
import { getUserRolesForApi, safeSessionStorageGetItem, waitForToken } from '@/lib/utils'
import type { RootState } from '@/store'
import { useAppDispatch, useAppSelector } from '@/store/hook'
import ScrollBtoIcon from '@/assets/svg/scrollBto.svg?react'
import { setIsAsking } from '@/store/chatSlice'
import SdreamLoading from '@/components/SdreamLoading'

export const Chat: React.FC = () => {
  const { id } = useParams<{ id: string }>()
  const location = useLocation()
  const [searchParams, setSearchParams] = useSearchParams()
  // 优先从 URL 查询参数读取 toolId（刷新后仍能保留），其次从 location.state 读取
  const rawToolIdFromUrl = searchParams.get('toolId')
  // 只有在非 SPA 导航（location.state 不存在）且链接携带 toolId 时才清空，避免影响站内点击历史记录
  const shouldForceClearUrlToolId = !location.state && Boolean(rawToolIdFromUrl)
  const toolIdFromUrl = shouldForceClearUrlToolId ? null : rawToolIdFromUrl
  // 添加调试日志，查看 location.state 的实际值
  const toolIdFromState = (location.state as { toolId?: string | null } | null)?.toolId
  // 优先使用 URL 中的 toolId，其次使用 state 中的 toolId
  const initialToolId = toolIdFromUrl !== null ? toolIdFromUrl : toolIdFromState
  const [isLoading, setIsLoading] = useState(false)
  const [allItems, setAllItems] = useState<ChatRecord[]>([])
  const dispatch = useAppDispatch()
  const { shouldSendQuestion, currentToolId, conversations } = useAppSelector((state: RootState) => state.conversation)
  // 标记历史记录是否已加载完成，用于确保自动发送问题在历史记录加载后执行
  const historyLoadedRef = useRef<{ conversationId: string | null }>({ conversationId: null })
  // 保存待发送的问题，用于在历史记录加载完成后发送
  const pendingQuestionRef = useRef<string | null>(null)
  const scrollableRef = useRef<HTMLDivElement | any>(null)
  const position = useScroll(scrollableRef)
  const currentIdRef = useRef<string | undefined>(id)
  const lastSentQuestionRef = useRef<string>('')
  const abortControllerRef = useRef<AbortController | null>(null)
  const [currentToolName, setCurrentToolName] = useState<string | undefined>(undefined)
  // 使用 ref 保存从 location.state 传递的 toolId，避免被异步操作覆盖
  const toolIdFromStateRef = useRef<string | null | undefined>(undefined)
  // 标记是否已通过 toolHistoryLoaded 事件加载数据，避免重复调用接口
  const toolHistoryLoadedRef = useRef<{ conversationId: string | null }>({ conversationId: null })
  // 标记是否已处理从收藏返回的逻辑，避免重复调用
  const fromCollectProcessedRef = useRef<boolean>(false)
  // 保存从收藏返回的标记，避免 location.state 被清除后丢失
  const fromCollectRef = useRef<boolean>(false)
  // 记录正在处理的 conversationId，避免重复调用
  const processingConversationIdRef = useRef<string | null>(null)

  // 当外部系统直接以 /chat/:id 链接进入（没有 location.state，且 URL 中也没有 toolId）时，
  // 视为一次新的会话入口：重置为制度活化，清除历史遗留的工具模式状态
  useEffect(() => {
    if (!location.state && !rawToolIdFromUrl) {
      dispatch(clearCurrentToolId())
      // sessionStorage.removeItem('currentToolId')
    }
  }, [location.state, rawToolIdFromUrl, dispatch])
  // 若链接中带有 toolId，进入页面后强制清空（避免沿用历史链接参数）
  useEffect(() => {
    if (!shouldForceClearUrlToolId)
      return

    // 1. 清除 Redux 中的 currentToolId
    dispatch(clearCurrentToolId())
    // 2. 清除 sessionStorage 中的 currentToolId
    // sessionStorage.removeItem('currentToolId')
    // 3. 清除 URL 中的 toolId 参数（如果存在）
    const currentUrl = new URL(window.location.href)
    if (currentUrl.searchParams.has('toolId')) {
      currentUrl.searchParams.delete('toolId')
      window.history.replaceState({}, '', currentUrl.toString())
    }
    // 4. 触发自定义事件，通知 ChatEditor 强制重置为制度活化
    window.dispatchEvent(new CustomEvent('forceResetToGeneralMode'))

    // 同步 react-router 的 searchParams 状态
    const newParams = new URLSearchParams(searchParams)
    newParams.delete('toolId')
    setSearchParams(newParams, { replace: true })
  }, [searchParams, setSearchParams, shouldForceClearUrlToolId, dispatch])
  // 调试用：展示当前页面链接与缓存中的 toolId
  const [debugInfo, setDebugInfo] = useState<{ href: string, sessionToolId: string | null }>({
    href: '',
    sessionToolId: null,
  })

  // 进入聊天页时，同步当前链接和缓存中的 toolId 到页面上展示
  useEffect(() => {
    try {
      const href = window.location.href
      const sessionToolId = sessionStorage.getItem('currentToolId')
      setDebugInfo({
        href,
        sessionToolId,
      })
    }
    catch (error) {
      console.error('同步页面链接与缓存信息失败:', error)
    }
  }, [])

  useEffect(() => {
    if (!debugInfo.href && !debugInfo.sessionToolId)
      return

    console.debug('[Chat] 当前链接 / 缓存 toolId:', debugInfo)
  }, [debugInfo])

  // 历史记录点击时将 toolId 通过路由 state 传入，优先使用该值快速同步高亮
  useEffect(() => {
    // 保存从 location.state 传递的 toolId 到 ref
    toolIdFromStateRef.current = initialToolId
    if (typeof initialToolId === 'undefined')
      return
    if (initialToolId) {
      // 统一转换为字符串，确保类型一致（真实API可能返回数字，需要转换为字符串）
      const normalizedToolId = String(initialToolId)
      dispatch(setCurrentToolId(normalizedToolId))
    }
    else {
      dispatch(clearCurrentToolId())
    }
  }, [dispatch, initialToolId])

  /** 处理正常stream的数据 */
  const handleStreamMesageData = (msg: any, question: string) => {
    setAllItems((prevItems) => {
      const newItems = [...prevItems] // 创建数组的浅拷贝
      const lastIndex = newItems.length - 1
      if (lastIndex >= 0) {
        // 创建最后一项的新对象，合并现有数据和新的 answer
        const originalAnswer = (newItems[lastIndex].answerList?.[0]?.answer || '') + (msg.content.data.answer ?? '\n')
        // 移除所有括号及其内容
        let filteredAnswer = originalAnswer.replace(/\([^)]*\)/g, '')
        // 去除 [参考文档《任意内容》 《任意内容》...] 格式的内容
        filteredAnswer = filteredAnswer.replace(/\[参考文档(?:《[^》]*》\s*)+\]/g, '')
        // 只移除空格和制表符，保留换行符（避免移除末尾的 \n\n）
        filteredAnswer = trimSpacesOnly(filteredAnswer)

        newItems[lastIndex] = {
          ...newItems[lastIndex],
          question,
          answerList: [
            {
              ...msg.content.data,
              isShow: false,
              answer: filteredAnswer,
            },
          ],
        }
      }
      return newItems
    })
  }

  /** 处理超过最大条数的数据 */
  const handleChatMaxCount = (msg: any, question: string) => {
    setAllItems((prevItems) => {
      const newItems = [...prevItems] // 创建数组的浅拷贝
      const lastIndex = newItems.length - 1
      if (lastIndex >= 0) {
        // 创建最后一项的新对象，合并现有数据和新的 answer
        newItems[lastIndex] = {
          ...newItems[lastIndex],
          question,
          answerList: [
            {
              ...msg.content.data,
              isShow: false,
              isChatMaxCount: true,
              endAnswerFlag: true,
              answer: '已达上限',
            },
          ],
        }
      }
      return newItems
    })
  }

  /** 提交问题 */
  const handleSubmitQuestion = async (question: string, productCode?: string, toolId?: string) => {
    // 优先读取缓存中的 toolId，再回退到传参或 Redux
    const sessionToolId = sessionStorage.getItem('currentToolId') ?? undefined
    const resolvedToolId = toolId ?? sessionToolId ?? currentToolId ?? undefined
    // 停止之前的请求
    if (abortControllerRef.current) {
      abortControllerRef.current.abort()
    }

    const isNew = allItems.length <= 1
    dispatch(setIsAsking(true))

    // 检查token
    await fetchCheckTokenApi()

    // 一次性添加用户问题和空的AI回答
    setAllItems(prevItems => [
      ...prevItems,
      {
        role: 'user',
        question,
      } as ChatRecord,
      {
        role: 'ai',
        answerList: [{ answer: '' }],
      } as ChatRecord,
    ])

    // 创建新的 AbortController
    abortControllerRef.current = new AbortController()

    let fetchUrl = `/conversation/api/conversation/mobile/v1/submit_question_stream`

    const viteOutputObj = import.meta.env.VITE_OUTPUT_OBJ || 'open'

    let proxy = ''
    if (viteOutputObj === 'open') {
      proxy = import.meta.env.MODE !== 'prod' ? '/api' : '/dev-sdream-api'
    }
 else {
      proxy = import.meta.env.MODE === 'dev' ? '/api' : '/dev-sdream-api'
    }

    fetchUrl = proxy + fetchUrl

    // 根据 toolId 添加相应的参数
    const requestBody: Record<string, any> = {
      question,
      conversationId: currentIdRef.current,
      stream: true,
      productCode,
      toolId: resolvedToolId ?? '',
    }

    // 制度活化：toolId 为空字符串或 undefined
    if (!resolvedToolId || resolvedToolId === '') {
      requestBody.toolId = ''
      requestBody.busiType = '01'
      requestBody.recordType = '01'
    }
    // 提质增效：toolId 为 6712395743241
    else if (resolvedToolId === '6712395743241') {
      requestBody.busiType = '01'
      requestBody.recordType = 'B00'
    }
    // 数据助手：toolId 为 6712395743240
    else if (resolvedToolId === '6712395743240') {
      requestBody.busiType = '01'
      requestBody.recordType = '01'
    }

    fetchStreamResponse(
      fetchUrl,
      requestBody,
      (msg) => {
        // 检查是否已被取消
        if (abortControllerRef.current?.signal.aborted) {
          return
        }

        // 处理错误
        if (msg?.type === 'ERROR') {
          // 出错时需要重置 isAsking 状态
          dispatch(setIsAsking(false))
          // 如果是 AbortError，不显示错误
          if (msg.content?.name === 'AbortError') {
            return
          }
          return
        }

        // 正常的stream数据
        if (msg?.type === 'DATA' && msg?.content?.code === '00000000') {
          console.log('========================流式输出====================', msg)

          console.log('========================流式输出字符串====================', msg.content?.data)
          handleStreamMesageData(msg, question)
        }
        if (msg?.type === 'DATA' && msg?.content?.code === '01010005') {
          handleChatMaxCount(msg, question)
          return
        }
        if (msg.type === 'END') {
          // 流式请求正常结束时重置 isAsking 状态
          dispatch(setIsAsking(false))
          if (isNew) {
            setTimeout(() => {
              dispatch(fetchConversations())
            }, 2000)
          }
        }
      },
      abortControllerRef.current.signal,
    )
  }

  /** 获取qa记录 */
  const getUserQaRecordPage = useCallback(async (conversationId: string, toolId?: string) => {
    // 检查是否正在处理相同的 conversationId，避免重复调用
    if (processingConversationIdRef.current === conversationId) {
      console.log('[Chat] 正在处理相同的 conversationId，跳过重复调用:', conversationId)
      return
    }

    setIsLoading(true)
    // 标记正在处理
    processingConversationIdRef.current = conversationId
    // 定义 finalConversationId 在 try 块外，以便 finally 块可以访问
    let finalConversationId = conversationId
    try {
      // 检测是否从收藏页返回（优先使用 ref 中保存的值，避免 location.state 被清除后丢失）
      // 增加 sessionStorage 检查，用于 navigate(-1) 的情况
      const fromCollect = fromCollectRef.current || Boolean(location.state?.fromCollect) || Boolean(sessionStorage.getItem('fromCollect'))

      // 如果从收藏页返回，先调用 get_user_conversation_session 获取新的会话ID
      if (fromCollect && !fromCollectProcessedRef.current) {
        fromCollectProcessedRef.current = true
        const sessionToolId = safeSessionStorageGetItem('currentToolId') || ''
        try {
          await waitForToken()
          const sessionRes = await fetchSessionConversationId({ busiId: sessionToolId })
          if (sessionRes?.data?.conversationId) {
            finalConversationId = sessionRes.data.conversationId
            console.log('[Chat] 从收藏返回 - 获取到新会话ID:', finalConversationId)
            // 如果获取到新会话ID，设置标记避免路由变化时重复调用
            toolHistoryLoadedRef.current.conversationId = finalConversationId
            // 更新 processingConversationIdRef 为新会话ID，避免使用旧ID重复调用
            processingConversationIdRef.current = finalConversationId
          }
        }
        catch (error) {
          console.error('[Chat] 从收藏返回 - 获取会话ID失败:', error)
          // 如果获取失败，继续使用原来的 conversationId
        }
      }

      // 如果从收藏页返回，触发事件通知 HomeNew 刷新问题列表
      if (fromCollect) {
        const eventToolId = toolId || currentToolId || safeSessionStorageGetItem('currentToolId') || ''
        console.log('[Chat] 检测到从收藏页返回 (fromCollect=true) - 准备触发 refreshQuestionsFromCollect 事件')
        console.log('[Chat] refreshQuestionsFromCollect - 传递的 toolId:', eventToolId)
        // 清除 sessionStorage 中的标记，避免影响后续路由
        sessionStorage.removeItem('fromCollect')
        // 延迟触发事件，确保 Home 组件的监听器已注册
        setTimeout(() => {
          window.dispatchEvent(new CustomEvent('refreshQuestionsFromCollect', {
            detail: {
              toolId: eventToolId,
            },
          }))
          console.log('[Chat] refreshQuestionsFromCollect 事件已触发')
        }, 100)
      }
      await waitForToken()
      const res = await fetchUserQaRecordPage(finalConversationId, toolId || safeSessionStorageGetItem('currentToolId') || '')
      console.log('qaRes chatEditor2222222', res)
      const qaRecords = res.data || []
      const messages = [{ role: 'system' } as ChatRecord, ...processApiResponse(qaRecords)]
      // 处理历史记录中的参考文档标记
      const processedMessages = messages.map((item) => {
        if (item.role === 'ai' && item.answerList?.[0]?.answer) {
          return {
            ...item,
            answerList: item.answerList.map(answerItem => ({
              ...answerItem,
              answer: answerItem.answer
                ?.replace(/\([^)]*\)/g, '')
                .replace(/\[参考文档(?:《[^》]*》\s*)+\]/g, '')
                .trim(),
            })),
          }
        }
        return item
      })
      setAllItems(processedMessages)
      // 优先从 qaRecords 中查找 toolId（这是实际使用的）
      const latestToolId = [...qaRecords].reverse().find(item => Boolean(item.toolId))?.toolId?.trim?.()
      const hasQaRecords = qaRecords.length > 0

      // 如果 qaRecords 中没有 toolId，尝试从 conversations 中查找当前会话的 toolId（会话级别）
      const conversationToolId = latestToolId || (conversations.find(conv => conv.conversationId === finalConversationId)?.toolId?.trim?.())

        // 优先使用从 location.state 传递的 toolId（如果存在），这是从历史记录点击传递过来的
      const toolIdFromState = toolIdFromStateRef.current !== undefined
        ? (toolIdFromStateRef.current ? String(toolIdFromStateRef.current) : null)
        : undefined

      // 确定最终使用的 toolId：优先使用从 location.state 传递的，其次使用 qaRecords 中的，最后使用 conversation 中的
      // 如果从 location.state 传递了 toolId，直接使用它（最高优先级）
      if (toolIdFromState !== undefined) {
        if (toolIdFromState) {
          // 只有当 Redux 中的 toolId 与最终确定的 toolId 不一致时，才更新
          if (currentToolId !== toolIdFromState) {
            dispatch(setCurrentToolId(toolIdFromState))
            // 从收藏返回时，同步到 sessionStorage，避免 ChatEditor 清除 toolId
            if (fromCollect) {
              sessionStorage.setItem('currentToolId', toolIdFromState)
            }
          }
          else {
            if (fromCollect && !sessionStorage.getItem('currentToolId')) {
              sessionStorage.setItem('currentToolId', toolIdFromState)
            }
          }
        }
        else {
          // 如果从 location.state 传递的是 null，清除 toolId
          if (currentToolId) {
            dispatch(clearCurrentToolId())
          }
        }
        // 清除 ref，避免下次路由变化时误用
        toolIdFromStateRef.current = undefined
      }
      else {
        // 如果没有从 location.state 传递 toolId（可能是嵌套路由传递失败），使用原来的逻辑
        // 优先使用 qaRecords 中的 toolId，其次使用 conversation 中的 toolId
        const finalToolId = latestToolId || conversationToolId || undefined

        if (finalToolId) {
          // 只有当 Redux 中的 toolId 与最终确定的 toolId 不一致时，才更新
          if (currentToolId !== finalToolId) {
            dispatch(setCurrentToolId(finalToolId))
            // 从收藏返回时，同步到 sessionStorage，避免 ChatEditor 清除 toolId
            if (fromCollect) {
              sessionStorage.setItem('currentToolId', finalToolId)
            }
          }
          else {
            // 从收藏返回时，确保 sessionStorage 中有值
            if (fromCollect && !sessionStorage.getItem('currentToolId')) {
              sessionStorage.setItem('currentToolId', finalToolId)
            }
          }
        }
        else {
          // 如果 qaRecords 和 conversation 中都没有 toolId
          // 如果有历史记录但没有 toolId，说明是制度活化，应该清除
          if (hasQaRecords && currentToolId) {
            dispatch(clearCurrentToolId())
          }
          // 如果没有历史记录，可能是新会话，但如果 Redux 中已经有 toolId（从 HistoryBarList 设置的），暂时保留
          // 因为可能是刚点击历史记录但 API 还没返回，或者 location.state 传递失败但 Redux 中已有正确的值
          else if (!hasQaRecords && currentToolId) {
            if (fromCollect && !sessionStorage.getItem('currentToolId')) {
              sessionStorage.setItem('currentToolId', currentToolId)
            }
          }
        }
      }
    }
    catch {
      // 错误处理
    }
    finally {
      setIsLoading(false)
      // 标记该会话的历史记录已加载完成
      historyLoadedRef.current.conversationId = finalConversationId
      console.log('[Chat] 历史记录加载完成:', {
        finalConversationId,
        currentIdRef: currentIdRef.current,
        pendingQuestion: pendingQuestionRef.current,
        lastSentQuestion: lastSentQuestionRef.current,
      })
      // 清除处理标记（支持新会话ID的情况）
      // finalConversationId 可能是新会话ID（从收藏返回时）或原来的 conversationId
      if (processingConversationIdRef.current === finalConversationId) {
        processingConversationIdRef.current = null
      }
      // 如果历史记录加载完成且有待发送的问题，触发自动发送
      if (pendingQuestionRef.current && finalConversationId === currentIdRef.current) {
        const questionToSend = pendingQuestionRef.current
        console.log('[Chat] 历史记录加载完成，检查待发送问题:', {
          questionToSend,
          lastSentQuestion: lastSentQuestionRef.current,
        })
        // 检查是否已经发送过相同的问题
        if (questionToSend !== lastSentQuestionRef.current) {
          console.log('[Chat] 触发待发送问题的自动发送:', questionToSend)
          lastSentQuestionRef.current = questionToSend
          pendingQuestionRef.current = null
          // 清除 shouldSendQuestion（如果还存在）
          dispatch(clearShouldSendQuestion())
          // 延迟发送，确保状态已更新
          setTimeout(() => {
            console.log('[Chat] 执行待发送问题的提交:', questionToSend)
            handleSubmitQuestion(questionToSend, undefined, currentToolId)
          }, 100)
        }
        else {
          console.log('[Chat] 待发送问题与已发送问题相同，跳过发送')
        }
      }
      else if (pendingQuestionRef.current) {
        console.log('[Chat] 有待发送问题但 conversationId 不匹配，等待:', {
          pendingQuestion: pendingQuestionRef.current,
          finalConversationId,
          currentIdRef: currentIdRef.current,
        })
      }
    }
  }, [dispatch, currentToolId, conversations, location.state])

  /** 点击滚动到底部 */
  const scrollToBottom = () => {
    scrollableRef.current.scrollTo(scrollableRef.current.scrollHeight, { behavior: 'smooth' })
  }

  useEffect(() => {
    if (id) {
      console.log('[Chat] 路由变化检查:', {
        id,
        currentIdRef: currentIdRef.current,
        locationState: location.state,
        toolHistoryLoadedId: toolHistoryLoadedRef.current.conversationId,
      })
      // 检查是否已通过 toolHistoryLoaded 事件加载过相同会话的数据
      // 如果已加载，跳过接口调用，避免重复请求
      if (toolHistoryLoadedRef.current.conversationId === id) {
        console.log('[Chat] 已通过 toolHistoryLoaded 加载，跳过历史记录接口调用')
        // 已加载，清除标记，避免影响后续正常的路由切换
        toolHistoryLoadedRef.current.conversationId = null
        // 更新 currentIdRef，确保状态一致
        currentIdRef.current = id
        return
      }

      // 检查是否正在通过工具切换加载历史记录（ChatEditor 正在处理）
      // 通过 sessionStorage 标记判断，避免时序问题导致的重复调用
      const toolHistoryLoadingId = sessionStorage.getItem('toolHistoryLoading')
      if (toolHistoryLoadingId === id) {
        console.log('[Chat] 检测到 toolHistoryLoading 标记，ChatEditor 正在加载历史记录，跳过重复调用')
        currentIdRef.current = id
        // 更新 toolId 相关状态（如果有）
        if (initialToolId !== undefined) {
          toolIdFromStateRef.current = initialToolId
        }
        return
      }

      // 检查是否跳过历史记录加载（点击常见问题时）
      const skipHistoryLoad = Boolean((location.state as { skipHistoryLoad?: boolean } | null)?.skipHistoryLoad)
      if (skipHistoryLoad) {
        console.log('[Chat] 检测到 skipHistoryLoad 标记，跳过历史记录加载，直接发送问题')
        currentIdRef.current = id
        // 更新 toolId 相关状态（如果有）
        if (initialToolId !== undefined) {
          toolIdFromStateRef.current = initialToolId
        }
        return
      }

      // 如果 id 没有变化，只是 location.state 变化，需要判断是否需要重新加载历史记录
      if (currentIdRef.current === id) {
        // 检查是否从收藏页返回（需要重新加载历史记录）
        const fromCollect = Boolean(location.state?.fromCollect) || Boolean(sessionStorage.getItem('fromCollect'))
        console.log('[Chat] conversationId 相同，检查是否需要重新加载:', {
          fromCollect,
          locationStateFromCollect: location.state?.fromCollect,
          sessionStorageFromCollect: sessionStorage.getItem('fromCollect'),
          allItemsLength: allItems.length,
        })
        // 如果 allItems 为空，说明是首次加载，需要加载历史记录
        if (!fromCollect && allItems.length > 0) {
          console.log('[Chat] 路由不变且非收藏页返回，跳过历史记录加载，只更新 toolId')
          // 不是从收藏页返回，只是 location.state 变化（比如点击常见问题但路由不变），
          // 不应该重新加载历史记录，只需要更新 toolId 相关状态
          // toolId 的更新逻辑在 useEffect 中处理（120-134行）
          return
        }
        console.log('[Chat] 从收藏页返回，即使 conversationId 相同也重新加载历史记录')
        // 如果是从收藏页返回，即使 conversationId 相同，也需要重新加载历史记录
        // 因为收藏页返回时可能会获取新的会话ID（在 getUserQaRecordPage 中处理）
      }

      // 停止之前的请求
      if (abortControllerRef.current) {
        abortControllerRef.current.abort()
        dispatch(setIsAsking(false))
      }

      console.log('[Chat] 开始加载历史记录，重置相关状态')
      currentIdRef.current = id
      lastSentQuestionRef.current = '' // 重置标记
      pendingQuestionRef.current = null // 清除待发送的问题
      // 重置历史记录加载标记，确保新会话时能正确触发自动发送
      historyLoadedRef.current.conversationId = null
      // 保存 fromCollect 标记到 ref，避免 location.state 被清除后丢失
      // 优先使用 location.state，其次使用 sessionStorage（用于 navigate(-1) 的情况）
      fromCollectRef.current = Boolean(location.state?.fromCollect) || Boolean(sessionStorage.getItem('fromCollect'))
      // 重置 fromCollect 处理标记，确保每次路由变化时都能正确处理
      fromCollectProcessedRef.current = false
      // 清除处理标记，允许新的 conversationId 处理
      processingConversationIdRef.current = null

      // 等待 token 就绪后再调用接口
      waitForToken().then(() => {
        console.log('[Chat] Token 就绪，调用 getUserQaRecordPage:', id)
        getUserQaRecordPage(id)
      })
    }
  }, [id, location.state])

  // 处理shouldSendQuestion的变化 - 自动发送问题
  useEffect(() => {
    console.log('[Chat] shouldSendQuestion 变化检查:', {
      shouldSendQuestion,
      currentIdRef: currentIdRef.current,
      isLoading,
      lastSentQuestion: lastSentQuestionRef.current,
      historyLoadedId: historyLoadedRef.current.conversationId,
      allItemsLength: allItems.length,
      pendingQuestion: pendingQuestionRef.current,
    })
    if (
      shouldSendQuestion
      && currentIdRef.current
      && !isLoading
      && shouldSendQuestion !== lastSentQuestionRef.current
    ) {
      // 如果当前会话已有历史记录（allItems.length > 1），需要等待历史记录加载完成
      // 如果历史记录已加载完成（historyLoadedRef.current.conversationId === currentIdRef.current），可以立即发送
      // 如果是新会话（allItems.length <= 1），可以立即发送
      const canSend = historyLoadedRef.current.conversationId === currentIdRef.current || allItems.length <= 1
      console.log('[Chat] 自动发送问题判断:', {
        canSend,
        historyLoaded: historyLoadedRef.current.conversationId === currentIdRef.current,
        isNewSession: allItems.length <= 1,
      })
      if (canSend) {
        const questionToSend = shouldSendQuestion
        console.log('[Chat] 可以立即发送，准备发送问题:', questionToSend)
        lastSentQuestionRef.current = questionToSend
        pendingQuestionRef.current = null
        // 立即清除shouldSendQuestion，防止重复发送
        dispatch(clearShouldSendQuestion())
        // 确保历史记录加载完成后再发送问题
        setTimeout(() => {
          console.log('[Chat] 执行自动发送问题:', questionToSend)
          handleSubmitQuestion(questionToSend, undefined, currentToolId)
        }, 100)
      }
      else {
        console.log('[Chat] 历史记录还在加载中，保存待发送问题到 pendingQuestionRef:', shouldSendQuestion)
        // 如果历史记录还在加载中，保存待发送的问题，等待历史记录加载完成后再发送
        pendingQuestionRef.current = shouldSendQuestion
      }
    }
  }, [shouldSendQuestion, isLoading, allItems.length, currentToolId, id])

  // 根据 currentToolId 获取对应的 toolName
  useEffect(() => {
    const getToolNameFromToolId = async () => {
      if (currentToolId) {
        try {
          // 优先从缓存读取工具列表，避免重复调用接口
          const userRoles = getUserRolesForApi()
          const cacheKey = `toolList_${JSON.stringify(userRoles)}`
          const cached = sessionStorage.getItem(cacheKey)

          if (cached) {
            try {
              const { toolList, timestamp } = JSON.parse(cached)
              // 缓存有效（5分钟内）
              if (Date.now() - timestamp < 5 * 60 * 1000) {
                const tool = toolList.find((t: any) => t.toolId === currentToolId)
                if (tool?.toolName) {
                  setCurrentToolName(tool.toolName)
                  return
                }
              }
            }
            catch {
              // 缓存解析失败，继续调用接口
            }
          }

          // 缓存无效或找不到工具时才调用接口
          await waitForToken()
          const res = await fetchToolList({ userRoles })
          if (res?.data) {
            const tool = res.data.find((t: any) => t.toolId === currentToolId)
            if (tool?.toolName) {
              setCurrentToolName(tool.toolName)
            }
          }
        }
        catch (error) {
          console.error('获取工具列表失败:', error)
        }
      }
      else {
        // 制度活化
        setCurrentToolName('制度活化')
      }
    }
    getToolNameFromToolId()
  }, [currentToolId])

  // 监听工具按钮点击事件，更新 ChatWelcome 提示语和 toolId
  useEffect(() => {
    const handleToolClickEvent = (event: CustomEvent) => {
      const { isToolBtn, toolId, toolName } = event.detail
      // 保存当前选择的 toolName
      setCurrentToolName(toolName)
      // 保存当前选择的 toolId 到 Redux
      if (!isToolBtn && toolId) {
        // 提质增效模式，保存 toolId
        dispatch(setCurrentToolId(toolId))
      }
      else {
        // 制度活化，清除 toolId
        dispatch(clearCurrentToolId())
      }
      // 历史记录加载已由 ChatEditor 通过 toolHistoryLoaded 事件处理，此处不再重复调用
    }
    window.addEventListener('toolButtonClick', handleToolClickEvent as EventListener)
    return () => {
      window.removeEventListener('toolButtonClick', handleToolClickEvent as EventListener)
    }
  }, [dispatch])

  // 监听工具历史记录加载事件，渲染历史会话
  useEffect(() => {
    const handleToolHistoryLoaded = (event: CustomEvent) => {
      const { conversationId, toolId, toolName, qaRecords } = event.detail
      setIsLoading(true)
      try {
        // 标记已通过事件加载数据，避免 useEffect 重复调用接口
        toolHistoryLoadedRef.current.conversationId = conversationId
        // 更新当前会话ID
        currentIdRef.current = conversationId
        // 处理历史记录数据
        const messages = [{ role: 'system' } as ChatRecord, ...processApiResponse(qaRecords)]
        // 处理历史记录中的参考文档标记
        const processedMessages = messages.map((item) => {
          if (item.role === 'ai' && item.answerList?.[0]?.answer) {
            return {
              ...item,
              answerList: item.answerList.map(answerItem => ({
                ...answerItem,
                answer: answerItem.answer
                  ?.replace(/\([^)]*\)/g, '')
                  .replace(/\[参考文档(?:《[^》]*》\s*)+\]/g, '')
                  .trim(),
              })),
            }
          }
          return item
        })
        setAllItems(processedMessages)
        // 更新 toolName
        if (toolName) {
          setCurrentToolName(toolName)
        }
        // 更新 toolId
        if (toolId) {
          dispatch(setCurrentToolId(toolId))
        }
      }
      catch (error) {
        console.error('处理历史记录失败:', error)
      }
      finally {
        setIsLoading(false)
        // 标记该会话的历史记录已加载完成（通过工具切换事件加载）
        historyLoadedRef.current.conversationId = conversationId
        // 清除 toolHistoryLoading 标记，避免影响后续路由切换
        const toolHistoryLoadingId = sessionStorage.getItem('toolHistoryLoading')
        if (toolHistoryLoadingId === conversationId) {
          sessionStorage.removeItem('toolHistoryLoading')
        }
      }
    }
    window.addEventListener('toolHistoryLoaded', handleToolHistoryLoaded as EventListener)
    return () => {
      window.removeEventListener('toolHistoryLoaded', handleToolHistoryLoaded as EventListener)
    }
  }, [dispatch])

  return (
    <div className={styles.scrollView}>
      <div className={`${styles.chatPage} relative`}>
        <div className={`${styles.content}`}>
          {isLoading && (
            <div className="w-full h-full flex justify-center items-center">
              <SdreamLoading />
            </div>
          )}
          {!isLoading && (
            <motion.div
              ref={scrollableRef}
              initial={{ opacity: 0, y: -10 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{
                duration: 0.3,
                opacity: { duration: 0.1 },
              }}
              className={`${styles.scrollable} scrollbar-hide scroll-smooth`}
            >
              <div className={styles.inter}>
                {allItems.map((record, index) => {
                  const recordId = record.answerList?.[0]?.recordId || record.groupId
                  const uniqueKey = recordId
                    ? `${record.role}-${recordId}`
                    : `${record.role}-${record.question || record.answerList?.[0]?.answer || ''}-${index}`
                  return (
                  <div
                    className="w-full chatItem mx-auto"
                    key={uniqueKey}
                  >
                    {record.role === 'system' && <ChatWelcome toolName={currentToolName} />}
                    {record.role === 'user' && <ChatItemUser record={record} />}
                    {record.role === 'ai' && (
                      <ChatAnswerBox
                        onSubmitQuestion={handleSubmitQuestion}
                        isLastAnswer={index === allItems.length - 1}
                        showIndex={0}
                        record={record}
                        index={index}
                      />
                    )}
                  </div>
                  )
                })}
              </div>
            </motion.div>
          )}
        </div>
        <div className="relative box-border px-[0] mx-auto iptContainer w-full flex-shrink-0 sm:px-0 pb-[18px]">
          <div className="absolute left-1/2 ml-[-20px] top-[-45px] sm:top-[-65px]">
            <motion.div
              initial="hidden"
              animate={(position?.top as number) < -20 ? 'visible' : 'hidden'}
              variants={{
                hidden: { opacity: 0, y: 20, pointerEvents: 'none' as const },
                visible: { opacity: 1, y: 0, pointerEvents: 'auto' as const },
              }}
              transition={{ duration: 0.3, ease: 'easeInOut' }}
            >
              <Button onPress={scrollToBottom} radius="full" isIconOnly color="primary">
                <ScrollBtoIcon />
              </Button>
            </motion.div>
          </div>
          <ChatEditor
            onSubmit={(question, toolId) => handleSubmitQuestion(question, undefined, toolId)}
            onToolClick={(isToolBtn, toolId, toolName, shouldChangeStyle, conversationId, isLoading) => {
            // 如果传递了 loading 状态，更新 isLoading
            if (isLoading !== undefined) {
              setIsLoading(isLoading)
            }
            // 发送自定义事件到父组件
            window.dispatchEvent(new CustomEvent('toolButtonClick', {
              detail: { isToolBtn, toolId, toolName, shouldChangeStyle, conversationId },
            }))
          }}
            placeholders={[]}
          />
          <div className="hidden sm:block w-full text-center mt-[12px] text-[#3333334d] text-[12px]">
            内容由AI模型生成，其准确性和完整性无法保证，仅供参考
          </div>
        </div>
      </div>
    </div>
  )
}
