/**
 * 只移除字符串开头和结尾的空格和制表符，保留换行符
 * 用于流式处理时保留末尾的 \n\n 换行符
 */
export function trimSpacesOnly(text: string): string {
  return text.replace(/^[ \t]+|[ \t]+$/g, '')
}

export function formatMarkdown(text: string): string {
  // 首先移除 ♪ 符号之后的所有文本
  // 使用 trimSpacesOnly 保留换行符，避免移除末尾的 \n\n
  let formattedText = text.split('♪')[0]
  formattedText = trimSpacesOnly(formattedText)

  // 处理换行
  // 先处理连续的换行符（\n\n 或更多），确保它们被保留为段落分隔
  // 然后将单独的 \n 替换为 Markdown 的硬换行（两个空格 + \n）
  // 注意：\n\n 在 Markdown 中表示段落分隔，应该被保留
  formattedText = formattedText.replace(/\n{3,}/g, '\n\n') // 将3个或更多连续换行符压缩为2个
  formattedText = formattedText.replace(/(?<!\n)\n(?!\n)/g, '  \n') // 单独的 \n 转换为硬换行

  // 处理代码块
  formattedText = formattedText.replace(/```(\w+)?\n([\s\S]*?)\n```/g, (match, language, code) => {
    return `\n\`\`\`${language || ''}\n${code.trim()}\n\`\`\`\n`
  })

  // 处理行内代码
  formattedText = formattedText.replace(/`([^`\n]+)`/g, '`$1`')

  // 处理列表
  formattedText = formattedText.replace(/^( *)[-*+] /gm, '$1- ')

  // 处理标题
  formattedText = formattedText.replace(/^(#{1,6}) /gm, '$1 ')

  // 处理粗体和斜体
  formattedText = formattedText.replace(/(\*\*|__)(.*?)\1/g, '**$2**')
  formattedText = formattedText.replace(/(\*|_)(.*?)\1/g, '*$2*')

  // 处理链接
  formattedText = formattedText.replace(/\[([^\]]+)\]\(([^)]+)\)/g, '[$1]($2)')

  return formattedText
}
