import type React from 'react'
import { motion } from 'framer-motion'
import { useEffect, useRef, useState } from 'react'
import { useNavigate } from 'react-router-dom'
import { useClickAway } from 'ahooks'
import styles from './Navbar.module.less'
import { NavBarItem } from './components/NavBarItem'
import { User } from './components/User'
import { clearNavigationFlag, createConversation } from '@/store/conversationSlice'
import type { WithAuthProps } from '@/auth/withAuth'
import { withAuth } from '@/auth/withAuth'
import { NAV_BAR_ITEMS } from '@/config/nav'
import { useAppDispatch, useAppSelector } from '@/store/hook'
import Logo from '@/assets/svg/logo.svg?react'
import { isMobile } from '@/utils'

interface NavbarProps {
  isHistoryVisible: boolean
  onSetHistoryVisible: (visible: boolean) => void
}

const NavbarBase: React.FC<NavbarProps & WithAuthProps> = ({ isHistoryVisible, checkAuth, onSetHistoryVisible }) => {
  const dispatch = useAppDispatch()
  const navigate = useNavigate()

  const { currentConversationId, shouldNavigateToNewConversation } = useAppSelector(state => state.conversation)

  const handleCreateConversation = () => {
    dispatch(createConversation({
      conversationData: {},
      shouldNavigate: true,
      shouldSendQuestion: '',
    }))
  }

  const [isH5NavVisible, setIsH5NavVisible] = useState(isMobile())

  const handleClick = (type: string | undefined) => {
    if (type === 'logo') {
      if (!isH5NavVisible) {
        navigate('/')
        setIsH5NavVisible(true)
      }
      else {
        setIsH5NavVisible(!isH5NavVisible)
      }
      return
    }
    setIsH5NavVisible(true)
    if (!checkAuth()) {
      return
    }

    if (type === 'add') {
      handleCreateConversation()
    }

    if (type === 'history') {
      onSetHistoryVisible(!isHistoryVisible)
    }

    if (type === 'collect') {
      navigate('/collect')
    }

    if (type === 'tools') {
      navigate('/tools')
    }
  }
  const handleLogout = () => {
    onSetHistoryVisible(false)
    navigate('/')
  }

  useEffect(() => {
    if (shouldNavigateToNewConversation && currentConversationId) {
      navigate(`/chat/${currentConversationId}`)
      dispatch(clearNavigationFlag())
    }
  }, [shouldNavigateToNewConversation, currentConversationId, navigate, dispatch])

  const navRef = useRef<HTMLButtonElement>(null)
  useClickAway(() => {
    setIsH5NavVisible(true)
  }, navRef)

  return (
    <motion.nav ref={navRef} className="h-full flex-shrink-0 flex flex-col items-center justify-center">
      {/* hidden */}
      <motion.div className={`${styles.layoutNavBarAgent}  rounded-full ${isH5NavVisible ? 'h-[55px] bg-white/40' : 'h-[380px]'} w-[54px] overflow-hidden py-[8px] sm:w-[64px] sm:bg-white gap-[24px] sm:h-auto sm:rounded-3xl sm:flex sm:py-[24px] sm:px-[8px]`}>
        <NavBarItem isHistoryVisible={isHistoryVisible} onClick={handleClick} icon={Logo} label="" key="logo" type={isMobile() ? 'logo3' : 'logo'} />

        {NAV_BAR_ITEMS.map((item) => {
          return (
            <NavBarItem isHistoryVisible={isHistoryVisible} onClick={handleClick} icon={item.icon} label={item.label} key={item.key} type={item.key} />
          )
        })}
        <User onLogout={handleLogout} />
      </motion.div>
    </motion.nav>
  )
}

export const Navbar = withAuth(NavbarBase)
