import { Button, Tooltip } from '@heroui/react'
import { useState } from 'react'
import { useDebounceFn } from 'ahooks'
import type { Answer } from '@/types/chat'
import LikeIcon from '@/assets/svg/zan.svg?react'
import UnLikeIcon from '@/assets/svg/cai.svg?react'
import CopyIcon from '@/assets/svg/copy.svg?react'
import CollectIcon from '@/assets/svg/shouc.svg?react'
import LikeIconA from '@/assets/svg/zanA.svg?react'
import UnLikeIconA from '@/assets/svg/caiA.svg?react'
import CollectIconA from '@/assets/svg/shoucA.svg?react'
import useToast from '@/hooks/useToast'
import { fetchCancelCollection, fetchSubmitCollection, fetchSubmitFeedback } from '@/api/chat'
import { UnLikeModal } from '@/components/UnLikeModal'

interface ChatAnswerOperateProps {
  answer: Answer
}
export const ChatAnswerOperate: React.FC<ChatAnswerOperateProps> = ({ answer }) => {
  const showToast = useToast()
  const [isCollect, setIsCollect] = useState(answer.collectionFlag)
  const [isLike, setIsLike] = useState(answer.feedbackStatus === '01')
  const [isUnLike, setIsUnLike] = useState(answer.feedbackStatus === '02')
  const [isOpenUnLikeModal, setIsOpenUnLikeOpen] = useState(false)

  const handleCopy = async () => {
    if (!navigator.clipboard) {
      showToast('您的浏览器不支持复制', 'error')
      return
    }
    await navigator.clipboard.writeText(answer.answer)

    showToast('复制成功！快去分享吧！', 'success')
  }

  const handleCollect = useDebounceFn(async () => {
    if (!isCollect) {
      setIsCollect(true)
      const res = await fetchSubmitCollection(answer.recordId || '')
      if (res.ok) {
        showToast('收藏成功！', 'success')
      }
    }
    else {
      setIsCollect(false)
      await fetchCancelCollection(answer.recordId || '')
    }
  }, { wait: 200 })

  const handleLike = useDebounceFn(async () => {
    if (!isLike) {
      setIsLike(true)
      setIsUnLike(false)
      await fetchSubmitFeedback({
        recordId: answer.recordId,
        feedbackStatus: '01',
        content: '',
      })
      showToast('感谢您的反馈', 'success', {
        icon: '🙏🏻',
      })
    }
    else {
      setIsLike(false)
      await fetchSubmitFeedback({
        recordId: answer.recordId,
        feedbackStatus: '00',
        content: '',
      })
    }
  })

  const handleUnLike = async () => {
    if (!isUnLike) {
      setIsOpenUnLikeOpen(true)
    }
    else {
      setIsUnLike(false)
      await fetchSubmitFeedback({
        recordId: answer.recordId,
        feedbackStatus: '00',
        content: '',
      })
    }
  }

  const handleClose = (isSubmit?: boolean) => {
    setIsOpenUnLikeOpen(false)
    if (isSubmit) {
      setIsLike(false)
      setIsUnLike(true)
      showToast('感谢您的反馈', 'success', {
        icon: '🙏🏻',
      })
    }
  }
  return (
    <div className="sm:mt-[12px] flex gap-[4px] justify-end">
      {/* 点赞 */}
      <Tooltip color="foreground" content={isLike ? '取消点赞' : '点赞'} className="capitalize">
        <Button variant="light" isIconOnly aria-label="LikeIcon" onPress={handleLike.run}>
          {isLike ? <LikeIconA /> : <LikeIcon />}
        </Button>
      </Tooltip>
      {/* 点踩 */}
      <Tooltip color="foreground" content={isUnLike ? '取消点踩' : '点踩'} className="capitalize">
        <Button variant="light" isIconOnly aria-label="UnLikeIcon" onPress={handleUnLike}>
          {isUnLike ? <UnLikeIconA /> : <UnLikeIcon />}
        </Button>
      </Tooltip>
      {/* 复制 */}
      <Tooltip color="foreground" content="复制" className="capitalize">
        <Button variant="light" isIconOnly aria-label="CopyIcon" onPress={handleCopy}><CopyIcon /></Button>
      </Tooltip>
      {/* 收藏 */}
      <Tooltip color="foreground" content={isCollect ? '取消收藏' : '收藏'} className="capitalize">
        <Button variant="light" isIconOnly aria-label="CollectIcon" onPress={handleCollect.run}>
          {isCollect ? <CollectIconA /> : <CollectIcon />}
        </Button>
      </Tooltip>
      {/* 重新生成 */}
      {/* <Tooltip color="foreground" content="重新生成" className="capitalize">
        <Button variant="light" isIconOnly aria-label="ReloadIcon"><ReloadIcon /></Button>
      </Tooltip> */}

      <UnLikeModal answer={answer} isOpen={isOpenUnLikeModal} onClose={handleClose} />
    </div>
  )
}
