import type React from 'react'
import { motion } from 'framer-motion'
import { useEffect, useState } from 'react'
import { useLocation } from 'react-router-dom'
import { Navbar } from '../Navbar'
import { HistoryBar } from '../HistoryBar/HistoryBar'
import styles from './MainLayout.module.less'
import { useAuth } from '@/auth/AuthContext'
import { LoginModal } from '@/components/LoginModal'
import { useAppDispatch } from '@/store/hook'
import { fetchConversations } from '@/store/conversationSlice'

interface MainLayoutProps {
  children: React.ReactNode
}

const contentVariants = {
  expanded: {
    width: '90px',
    transition: { type: 'spring', stiffness: 300, damping: 30 },
  },
  shrunk: {
    width: '340px',
    transition: { type: 'spring', stiffness: 300, damping: 30 },
  },
}

export const MainLayout: React.FC<MainLayoutProps> = ({ children }) => {
  const { showLoginModal, toggleLoginModal } = useAuth()
  const [isHistoryVisible, setHistoryVisible] = useState(false)
  const location = useLocation()
  const dispatch = useAppDispatch()

  useEffect(() => {
    dispatch(fetchConversations())
  }, [dispatch])

  useEffect(() => {
    if (location.pathname === '/tools' || location.pathname === '/collect') {
      setHistoryVisible(false)
    }
  }, [location.pathname])

  return (
    <motion.main className={styles.layoutMain}>
      <motion.div
        animate={isHistoryVisible ? 'shrunk' : 'expanded'}
        variants={contentVariants}
        className={`hidden sm:flex h-full pl-[12px] items-center ${isHistoryVisible ? 'w-[340px]' : 'w-[90px]'}`}
      >
        <Navbar isHistoryVisible={isHistoryVisible} onSetHistoryVisible={setHistoryVisible} />
        <HistoryBar isVisible={isHistoryVisible} />
      </motion.div>
      <motion.div
        className={`${styles.layoutContent}`}
      >
        {children}
      </motion.div>

      <LoginModal isOpen={showLoginModal} onClose={toggleLoginModal} />
    </motion.main>
  )
}
