import type React from 'react'
import { motion } from 'framer-motion'
import { Button, Tooltip } from '@nextui-org/react'
import { useLocalStorageState, useToggle } from 'ahooks'
import { useEffect, useRef } from 'react'
import styles from './Navbar.module.less'
import { NavBarItem } from './components/NavBarItem'
import Logo from '@/assets/svg/logo.svg?react'
import AddNewChat from '@/assets/svg/addNewChat.svg?react'
import HistoryChat from '@/assets/svg/historyChat.svg?react'
import Collect from '@/assets/svg/collect.svg?react'
import Tools from '@/assets/svg/tools.svg?react'
import UserIcon from '@/assets/svg/user.svg?react'
import { LoginModal } from '@/components/LoginModal'
import type { WithAuthProps } from '@/auth/withAuth'
import { withAuth } from '@/auth/withAuth'

const NAV_BAR_ITEMS = [
  { icon: Logo, label: '', key: 'logo' },
  { icon: '', label: '', key: 'line1' },
  { icon: AddNewChat, label: '新建对话', key: 'add' },
  { icon: HistoryChat, label: '历史对话', key: 'history' },
  { icon: Collect, label: '收藏', key: 'collect' },
  { icon: '', label: '', key: 'line2' },
  { icon: Tools, label: '工具', key: 'tools' },
  { icon: '', label: '', key: 'line3' },
]

interface NavbarProps {
  isCollapsed: boolean
  onToggle: () => void
}

const NavbarBase: React.FC<NavbarProps & WithAuthProps> = ({ checkAuth, onToggle }) => {
  const [token] = useLocalStorageState<string | undefined>('__TOKEN__', {
    defaultValue: '',
    listenStorageChange: true,
  })
  const [isOpenLogTip, isOpenLogTipActions] = useToggle()

  const [isOpenLoginModal, isOpenLoginModalActions] = useToggle()

  const intervalRef = useRef<NodeJS.Timeout | null>(null)
  const handleShowLoginTip = () => {
    intervalRef.current = setTimeout(() => {
      if (!token) {
        isOpenLogTipActions.setRight()
      }
      if (intervalRef.current) {
        clearInterval(intervalRef.current)
        intervalRef.current = null
      }
    }, 1000)
  }

  const handleCloseLoginModal = () => {
    isOpenLoginModalActions.setLeft()
    handleShowLoginTip()
  }

  const handleClick = (type: string | undefined) => {
    if (!checkAuth())
      return

    if (type === 'history') {
      onToggle()
    }
  }

  useEffect(() => {
    handleShowLoginTip()
  }, [])

  return (
    <>
      <motion.nav className="h-full flex-shrink-0 flex flex-col items-center justify-center">
        <motion.div className={`${styles.layoutNavBarAgent} sm:flex hidden w-[64px] bg-white gap-[24px]`}>
          {NAV_BAR_ITEMS.map((item) => {
            return (
              <NavBarItem onClick={handleClick} icon={item.icon} label={item.label} key={item.key} type={item.key} />
            )
          })}
          <Tooltip isOpen={isOpenLogTip} color="foreground" content="登录体验更多功能" placement="right">
            <Button onClick={checkAuth} variant="light" isIconOnly aria-label="Like">
              <UserIcon />
            </Button>
          </Tooltip>
        </motion.div>
      </motion.nav>
      <LoginModal onClose={handleCloseLoginModal} isOpen={isOpenLoginModal} />
    </>
  )
}

export const Navbar = withAuth(NavbarBase)
