// AuthContext.tsx
import type { ReactNode } from 'react'
import React, { createContext, useContext, useState } from 'react'
import { useLocalStorageState } from 'ahooks'

interface AuthContextType {
  isLoggedIn: boolean
  showLoginModal: boolean
  login: () => void
  logout: () => void
  toggleLoginModal: () => void
}

const AuthContext = createContext<AuthContextType | undefined>(undefined)

interface AuthProviderProps {
  children: ReactNode
}

export const AuthProvider: React.FC<AuthProviderProps> = ({ children }) => {
  const [token] = useLocalStorageState<string | undefined>(
    '__TOKEN__',
    {
      defaultValue: '',
    },
  )
  const [isLoggedIn, setIsLoggedIn] = useState(Boolean(token))
  const [showLoginModal, setShowLoginModal] = useState(false)

  const login = () => {
    setIsLoggedIn(true)
    setShowLoginModal(false)
  }

  const logout = () => {
    setIsLoggedIn(false)
  }

  const toggleLoginModal = () => {
    setShowLoginModal(!showLoginModal)
  }

  return (
    // eslint-disable-next-line react/no-unstable-context-value
    <AuthContext.Provider value={{ isLoggedIn, showLoginModal, login, logout, toggleLoginModal }}>
      {children}
    </AuthContext.Provider>
  )
}

// eslint-disable-next-line react-refresh/only-export-components
export function useAuth(): AuthContextType {
  const context = useContext(AuthContext)
  if (context === undefined) {
    throw new Error('useAuth must be used within an AuthProvider')
  }
  return context
}
