import type React from 'react'
import { motion } from 'framer-motion'
import { useEffect, useRef, useState } from 'react'
import { useLocation, useNavigate } from 'react-router-dom'
import { useClickAway, useSessionStorageState } from 'ahooks'
import styles from './Navbar.module.less'
import { NavBarItem } from './components/NavBarItem'
import { clearNavigationFlag, createConversation } from '@/store/conversationSlice'
import type { WithAuthProps } from '@/auth/withAuth'
import { withAuth } from '@/auth/withAuth'
import { NAV_BAR_ITEMS } from '@/config/nav'
import { useAppDispatch, useAppSelector } from '@/store/hook'
import { isMobile } from '@/utils'
// import { User } from './components/User'

interface NavbarProps {
  isHistoryVisible: boolean
  onSetHistoryVisible: (visible: boolean) => void
}

const NavbarBase: React.FC<NavbarProps & WithAuthProps> = ({ isHistoryVisible, checkAuth, onSetHistoryVisible }) => {
  const dispatch = useAppDispatch()
  const navigate = useNavigate()
  const location = useLocation()

  const { currentConversationId, shouldNavigateToNewConversation, currentToolId } = useAppSelector(state => state.conversation)

  const handleCreateConversation = () => {
    dispatch(createConversation({
      conversationData: {},
      shouldNavigate: true,
      shouldSendQuestion: '',
    }))
  }

  const [isH5NavVisible, setIsH5NavVisible] = useState(isMobile())

  const handleClick = (type: string | undefined) => {
    if (type === 'logo') {
      if (!isH5NavVisible) {
        navigate('/')
        setIsH5NavVisible(true)
      }
      else {
        setIsH5NavVisible(!isH5NavVisible)
      }
      return
    }
    setIsH5NavVisible(true)
    if (!checkAuth()) {
      return
    }

    if (type === 'add') {
      if (location.pathname.includes('/chat')) {
        handleCreateConversation()
      }
      else {
        navigate('/')
      }
      onSetHistoryVisible(false)
    }

    if (type === 'history') {
      onSetHistoryVisible(!isHistoryVisible)
    }

    if (type === 'collect') {
      // 从URL中提取会话ID并存储到 sessionStorage
      const pathParts = location.pathname.split('/')
      // 从路径中提取会话ID，获取 chat/ 后面的值
      // 查找 'chat' 在路径中的位置，然后获取其后一项作为会话ID
      const chatIndex = pathParts.indexOf('chat')
      const conversationId = chatIndex !== -1 && pathParts.length > chatIndex + 1
        ? pathParts[chatIndex + 1]
        : null

      if (conversationId) {
        sessionStorage.setItem('currentConversationId', conversationId)
      }
      navigate('/collect')
    }

    if (type === 'tools') {
      navigate('/tools')
    }
  }
  // const handleLogout = () => {
  //   onSetHistoryVisible(false)
  //   navigate('/')
  // }

  const [navBarVisibleLocal, setNavBarVisibleLocal] = useSessionStorageState<string | undefined>(
    '__NAV_BAR_VISIBLE_LOCAL__',
    {
      defaultValue: '0',
      listenStorageChange: true,
    },
  )

  const toggleNavBarVisible = () => {
    setNavBarVisibleLocal(navBarVisibleLocal === '1' ? '0' : '1')
  }

  useEffect(() => {
    if (shouldNavigateToNewConversation && currentConversationId) {
      // 获取当前 URL 的查询参数，保留除 'from' 之外的所有参数
      const currentSearchParams = new URLSearchParams(location.search)
      const paramsToKeep = new URLSearchParams()

      // 遍历当前查询参数，过滤掉 'from' 参数
      currentSearchParams.forEach((value, key) => {
        if (key !== 'from') {
          paramsToKeep.set(key, value)
        }
      })

      // 如果当前有选中的 toolId，添加到查询参数中
      if (currentToolId) {
        paramsToKeep.set('toolId', currentToolId)
      }

      // 构建跳转 URL
      const queryString = paramsToKeep.toString()
      const url = queryString
        ? `/chat/${currentConversationId}?${queryString}`
        : `/chat/${currentConversationId}`

      // 通过 location.state 传递 toolId，避免在 Chat 页面被误判为"外链残留参数"而强制清空
      navigate(url, {
        state: {
          toolId: currentToolId || null,
        },
      })
      dispatch(clearNavigationFlag())
    }
  }, [shouldNavigateToNewConversation, currentConversationId, currentToolId, navigate, dispatch, location.search])

  // keep latest conversation id in sessionStorage for cross-page returns (e.g., from collect)
  useEffect(() => {
    if (currentConversationId) {
      sessionStorage.setItem('currentConversationId', currentConversationId)
    }
  }, [currentConversationId])

  const navRef = useRef<HTMLButtonElement>(null)
  useClickAway(() => {
    setIsH5NavVisible(true)
  }, navRef)

  return (
    <>
      <motion.nav
        ref={navRef}
        animate={navBarVisibleLocal === '0' ? 'hidden' : ''}
        className="h-full flex-shrink-0 flex flex-col items-center justify-center relative"
      >
        {/* hidden */}
        <motion.div className={`layoutNavBarJoyride ${styles.layoutNavBarAgent}  rounded-full ${isH5NavVisible ? 'h-[55px] bg-white/70 shadow-md' : 'h-[380px]'} w-[54px] overflow-hidden py-[8px] sm:w-[64px] sm:bg-white gap-[24px] sm:h-auto sm:rounded-3xl sm:flex sm:py-[24px] sm:px-[8px]`}>
          {/* <NavBarItem isHistoryVisible={isHistoryVisible} onClick={handleClick} icon={Logo} label="" key="logo" type={isMobile() ? 'logo3' : 'logo'} /> */}

          {NAV_BAR_ITEMS.map((item) => {
            return (
              <NavBarItem isHistoryVisible={isHistoryVisible} onClick={handleClick} icon={item.icon} label={item.label} key={item.key} type={item.key} />
            )
          })}
          {/* <User onLogout={handleLogout} /> */}
        </motion.div>
        <div onClick={toggleNavBarVisible} className={`${styles.sidebarAction} ${navBarVisibleLocal === '0' ? styles.open : ''}`}></div>
      </motion.nav>
      {/* <JoyrideStep /> */}
    </>
  )
}

export const Navbar = withAuth(NavbarBase)
