import React, { useCallback, useEffect, useRef, useState } from 'react'
import { useParams } from 'react-router-dom'
import { Button } from '@heroui/react'
import { motion } from 'framer-motion'
import { useScroll } from 'ahooks'
import styles from './Chat.module.less'
import { processApiResponse } from './helper'
import { ChatWelcome } from './components/ChatWelcome'
import { ChatItemUser } from './components/ChatItem/ChatItemUser'
import { ChatAnswerBox } from './components/ChatItem/ChatAnswerBox'
import { ChatEditor } from '@/components/ChatEditor'
import type { ChatRecord } from '@/types/chat'
import { fetchUserQaRecordPage } from '@/api/conversation'
import { fetchCheckTokenApi, fetchStreamResponse } from '@/api/chat'
import { clearShouldSendQuestion, fetchConversations } from '@/store/conversationSlice'
import type { RootState } from '@/store'
import { useAppDispatch, useAppSelector } from '@/store/hook'
import ScrollBtoIcon from '@/assets/svg/scrollBto.svg?react'
import { setIsAsking } from '@/store/chatSlice'
import SdreamLoading from '@/components/SdreamLoading'

export const Chat: React.FC = () => {
  const { id } = useParams<{ id: string }>()
  const [isLoading, setIsLoading] = useState(false)
  const [allItems, setAllItems] = useState<ChatRecord[]>([])
  const dispatch = useAppDispatch()
  const { shouldSendQuestion } = useAppSelector((state: RootState) => state.conversation)
  const scrollableRef = useRef<HTMLDivElement | any>(null)
  const position = useScroll(scrollableRef)
  const currentIdRef = useRef<string | undefined>(id)
  const lastSentQuestionRef = useRef<string>('')
  const abortControllerRef = useRef<AbortController | null>(null)

  /** 处理正常stream的数据 */
  const handleStreamMesageData = (msg: any, question: string) => {
    setAllItems((prevItems) => {
      const newItems = [...prevItems] // 创建数组的浅拷贝
      const lastIndex = newItems.length - 1
      if (lastIndex >= 0) {
        // 创建最后一项的新对象，合并现有数据和新的 answer
        const originalAnswer = (newItems[lastIndex].answerList?.[0]?.answer || '') + msg.content.data.answer
        // 去除 [参考文档《任意内容》 《任意内容》...] 格式的内容
        const filteredAnswer = originalAnswer.replace(/\[参考文档(?:《[^》]*》\s*)+\]/g, '').trim()

        newItems[lastIndex] = {
          ...newItems[lastIndex],
          question,
          answerList: [
            {
              ...msg.content.data,
              isShow: false,
              answer: filteredAnswer,
            },
          ],
        }
      }
      return newItems
    })
  }

  /** 处理超过最大条数的数据 */
  const handleChatMaxCount = (msg: any, question: string) => {
    // toast(t => (
    //   <div className="flex items-center">
    //     <p className="text-[14px]">⚠️ 超过最大轮数上限！请新建对话 👉🏻</p>
    // <Button
    //   color="primary"
    //   size="sm"
    //   variant="light"
    //   isIconOnly
    //   onClick={() => {
    //     dispatch(createConversation({
    //       conversationData: {},
    //       shouldNavigate: true,
    //       shouldSendQuestion: '',
    //     }))
    //     toast.dismiss(t.id)
    //   }}
    // >
    //   <AddNewChat />
    // </Button>
    //   </div>
    // ), {
    //   position: 'bottom-center',
    //   duration: 0,
    //   style: {
    //     marginBottom: '120px',
    //   },
    // })
    setAllItems((prevItems) => {
      const newItems = [...prevItems] // 创建数组的浅拷贝
      const lastIndex = newItems.length - 1
      if (lastIndex >= 0) {
        // 创建最后一项的新对象，合并现有数据和新的 answer
        newItems[lastIndex] = {
          ...newItems[lastIndex],
          question,
          answerList: [
            {
              ...msg.content.data,
              isShow: false,
              isChatMaxCount: true,
              endAnswerFlag: true,
              answer: '已达上限',
            },
          ],
        }
      }
      return newItems
    })
  }

  /** 提交问题 */
  const handleSubmitQuestion = async (question: string, productCode?: string) => {
    // 停止之前的请求
    if (abortControllerRef.current) {
      abortControllerRef.current.abort()
    }

    const isNew = allItems.length <= 1
    dispatch(setIsAsking(true))

    // 检查token
    await fetchCheckTokenApi()

    // 一次性添加用户问题和空的AI回答
    setAllItems(prevItems => [
      ...prevItems,
      {
        role: 'user',
        question,
      } as ChatRecord,
      {
        role: 'ai',
        answerList: [{ answer: '' }],
      } as ChatRecord,
    ])

    // 创建新的 AbortController
    abortControllerRef.current = new AbortController()

    let fetchUrl = `/conversation/api/conversation/mobile/v1/submit_question_stream`
    const proxy = import.meta.env.MODE !== 'prod' ? '/api' : '/dev-sdream-api'
    fetchUrl = proxy + fetchUrl

    fetchStreamResponse(
      fetchUrl,
      {
        question,
        conversationId: currentIdRef.current,
        stream: true,
        productCode,
      },
      (msg) => {
        // 检查是否已被取消
        if (abortControllerRef.current?.signal.aborted) {
          return
        }

        // 处理错误
        if (msg?.type === 'ERROR') {
          // 如果是 AbortError，不显示错误
          if (msg.content?.name === 'AbortError') {
            return
          }
          return
        }

        // 正常的stream数据
        if (msg?.type === 'DATA' && msg?.content?.code === '00000000') {
          handleStreamMesageData(msg, question)
        }
        if (msg?.type === 'DATA' && msg?.content?.code === '01010005') {
          handleChatMaxCount(msg, question)
          return
        }
        if (msg.type === 'END') {
          if (isNew) {
            setTimeout(() => {
              dispatch(fetchConversations())
            }, 2000)
          }
        }
      },
      abortControllerRef.current.signal,
    )
  }

  /** 获取qa记录 */
  const getUserQaRecordPage = useCallback(async (conversationId: string) => {
    setIsLoading(true)
    try {
      const res = await fetchUserQaRecordPage(conversationId)
      const messages = [{ role: 'system' } as ChatRecord, ...processApiResponse(res.data)]
      // 处理历史记录中的参考文档标记
      const processedMessages = messages.map((item) => {
        if (item.role === 'ai' && item.answerList?.[0]?.answer) {
          return {
            ...item,
            answerList: item.answerList.map(answerItem => ({
              ...answerItem,
              answer: answerItem.answer?.replace(/\[参考文档(?:《[^》]*》\s*)+\]/g, '').trim(),
            })),
          }
        }
        return item
      })
      setAllItems(processedMessages)
    }
 catch {
      // 错误处理
    }
 finally {
      setIsLoading(false)
    }
  }, [])

  /** 点击滚动到底部 */
  const scrollToBottom = () => {
    scrollableRef.current.scrollTo(scrollableRef.current.scrollHeight, { behavior: 'smooth' })
  }

  useEffect(() => {
    if (id) {
      // 停止之前的请求
      if (abortControllerRef.current) {
        abortControllerRef.current.abort()
        dispatch(setIsAsking(false))
      }

      currentIdRef.current = id
      lastSentQuestionRef.current = '' // 重置标记
      getUserQaRecordPage(id)
    }
  }, [id])

  // 处理shouldSendQuestion的变化 - 自动发送问题
  useEffect(() => {
    if (
      shouldSendQuestion
      && currentIdRef.current
      && !isLoading
      && shouldSendQuestion !== lastSentQuestionRef.current
    ) {
      lastSentQuestionRef.current = shouldSendQuestion
      // 立即清除shouldSendQuestion，防止重复发送
      dispatch(clearShouldSendQuestion())
      // 确保历史记录加载完成后再发送问题
      setTimeout(() => {
        handleSubmitQuestion(shouldSendQuestion)
      }, 100)
    }
  }, [shouldSendQuestion, isLoading])

  return (
    <div className={styles.scrollView}>
      <div className={`${styles.chatPage} relative`}>
        {/* <ChatSlogan />
        <ChatMaskBar /> */}
        <div className={`${styles.content}`}>
          {isLoading && (
            <div className="w-full h-full flex justify-center items-center">
              <SdreamLoading />
            </div>
          )}
          {!isLoading && (
            <motion.div
              ref={scrollableRef}
              initial={{ opacity: 0, y: -10 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{
                duration: 0.3,
                opacity: { duration: 0.1 },
              }}
              className={`${styles.scrollable} scrollbar-hide scroll-smooth`}
            >
              <div className={styles.inter}>
                {allItems.map((record, index) => (
                  <div
                    className="w-full chatItem mx-auto"
                    key={`${record.role}-${record.id || index}-${
                      record.question || record.answerList?.[0]?.answer || ''
                    }`}
                  >
                    {record.role === 'system' && <ChatWelcome />}
                    {record.role === 'user' && <ChatItemUser record={record} />}
                    {record.role === 'ai' && (
                      <ChatAnswerBox
                        onSubmitQuestion={handleSubmitQuestion}
                        isLastAnswer={index === allItems.length - 1}
                        showIndex={0}
                        record={record}
                        index={index}
                      />
                    )}
                  </div>
                ))}
              </div>
            </motion.div>
          )}
        </div>
        <div className="relative box-border px-[0] mx-auto iptContainer w-full flex-shrink-0 sm:px-0 pb-[18px]">
          <div className="absolute left-1/2 ml-[-20px] top-[-45px] sm:top-[-65px]">
            <motion.div
              initial="hidden"
              animate={(position?.top as number) < -20 ? 'visible' : 'hidden'}
              variants={{
                hidden: { opacity: 0, y: 20, pointerEvents: 'none' as const },
                visible: { opacity: 1, y: 0, pointerEvents: 'auto' as const },
              }}
              transition={{ duration: 0.3, ease: 'easeInOut' }}
            >
              <Button onPress={scrollToBottom} radius="full" isIconOnly color="primary">
                <ScrollBtoIcon />
              </Button>
            </motion.div>
          </div>

          <ChatEditor onSubmit={handleSubmitQuestion} placeholders={[]} />
          <div className="hidden sm:block w-full text-center mt-[12px] text-[#3333334d] text-[12px]">
            内容由AI模型生成，其准确性和完整性无法保证，仅供参考
          </div>
        </div>
      </div>
    </div>
  )
}
