import type React from 'react'
import { useCallback, useEffect, useRef, useState } from 'react'
import { motion } from 'framer-motion'
import { Outlet, useLocation } from 'react-router-dom'
import { useLocalStorageState } from 'ahooks'
import styles from './Home.module.less'
import { QuestionList } from './components/QuestionList'
import HomeIcon1 from '@/assets/homeIcon1.png'
import HomeIcon2 from '@/assets/homeIcon2.png'
import SmartIce from '@/assets/smart-ice.png'
import { createConversation, fetchConversations } from '@/store/conversationSlice'
import { useAppDispatch } from '@/store/hook'
import { fetchEfficiencyQuestionList, fetchQuestionList } from '@/api/home'
import SdreamLoading from '@/components/SdreamLoading'
import { fetchLoginByToken, fetchLoginByUid } from '@/api/common'

function getAnimationProps(delay: number) {
  return {
    variants: {
      hidden: {
        opacity: 0,
        y: 50,
        scale: 0.9,
        rotateX: -6,
      },
      visible: {
        opacity: 1,
        y: 0,
        scale: 1,
        rotateX: 0,
        transition: {
          duration: 0.4,
          delay: delay * 0.1,
          ease: [0.25, 0.1, 0.25, 1],
        },
      },
    },
    initial: 'hidden',
    animate: 'visible',
  }
}

export const Home: React.FC = () => {
  const viteOutputObj = import.meta.env.VITE_OUTPUT_OBJ || 'open'
  const [isLoading, setIsLoading] = useState(false)
  const [isDataLoaded, setIsDataLoaded] = useState(false)
  const dispatch = useAppDispatch()
  const location = useLocation()
  const hasFetched = useRef(false)
  // 使用 useState
  const [productQuestions, setProductQuestions] = useState<any>({ content: [] })
  const [otherQuestions, setOtherQuestions] = useState<any>({ content: [] })
  const [isToolBtnActive, setIsToolBtnActive] = useState<boolean>(false)
  // 保存原始的configType为07的数据
  const [originalOtherQuestions, setOriginalOtherQuestions] = useState<any>({ content: [] })

  const [token, setToken] = useLocalStorageState<string | undefined>('__TOKEN__', {
    defaultValue: '',
  })

  /** 获取qa记录 */
  const getQuestionList = useCallback(async () => {
    setIsLoading(true)
    try {
      const param = {
        configTypeList: ['06', '07'],
      }
      const res = await fetchQuestionList(param)
      if (res && res.data) {
        for (let index = 0; index < res.data.length; index++) {
          const element = res.data[index]
          if (element.configType === '06') {
            element.content = JSON.parse(element.content)
            setProductQuestions(element)
          }
          if (element.configType === '07') {
            element.content = JSON.parse(element.content)
            setOtherQuestions(element)
            setOriginalOtherQuestions(element) // 保存原始数据
          }
        }
      }
    }
 catch (error) {
      // 可以在这里添加错误处理逻辑
      console.error('Failed to fetch chat records:', error)
    }
finally {
      setIsLoading(false)
      setIsDataLoaded(true)
    }
  }, [])

  const initConversation = () => {
    const fromCollect = location.state?.fromCollect
    // 只有在访问首页时才创建新对话，如果已经在聊天页面则不创建
    if (!fromCollect && (location.pathname === '/' || location.pathname === '/home')) {
      dispatch(
        createConversation({
          conversationData: {},
          shouldNavigate: true,
          shouldSendQuestion: '',
        }),
      )
    }
    // 清除状态以避免下次影响
    if (location.state?.fromCollect) {
      // 使用 replace 替换当前历史记录，清除 state
      window.history.replaceState({}, document.title, window.location.pathname)
    }
  }

  // 处理工具按钮点击
  const _handleToolClick = useCallback(async (isToolBtn: boolean, toolId?: string) => {
    if (!isToolBtn && toolId) {
      // 提质增效模式：只修改左侧页面内容，加载工具相关问题
      // 先清空数据，确保显示空数据样式
      setOtherQuestions((prev: any) => ({
        ...prev,
        content: [],
      }))
      setIsDataLoaded(false) // 重置加载状态
      try {
        const res = await fetchEfficiencyQuestionList({ toolId })
        if (res && res.data && res.data.questions) {
          setOtherQuestions((prev: any) => ({
            ...prev,
            content: res.data.questions || [],
          }))
        }
      }
      catch (error) {
        console.error('获取工具相关问题失败:', error)
      }
      finally {
        setIsDataLoaded(true) // 无论成功失败都标记为已加载
      }
    }
    else if (isToolBtn) {
      // 通用模式：恢复刷新时的状态，包括左侧内容也要恢复到初始状态
      setOtherQuestions(originalOtherQuestions)
      setIsDataLoaded(true) // 恢复原始数据时标记为已加载
    }
  }, [originalOtherQuestions])

  // 监听工具按钮点击事件
  useEffect(() => {
    const handleToolClickEvent = (event: CustomEvent) => {
      const { isToolBtn, toolId } = event.detail
      setIsToolBtnActive(isToolBtn)
      _handleToolClick(isToolBtn, toolId)
    }
    window.addEventListener('toolButtonClick', handleToolClickEvent as EventListener)
    return () => {
      window.removeEventListener('toolButtonClick', handleToolClickEvent as EventListener)
    }
  }, [_handleToolClick])

  const login = useCallback(async () => {
    // 防止重复调用
    if (hasFetched.current) {
      return
    }
    hasFetched.current = true
    const url = new URL(window.location.href)
    // 获取查询参数
    const searchParams = new URLSearchParams(url.search)
    const _loginCode = searchParams.get('loginCode')
    let res = {} as any
    if (viteOutputObj === 'inner') {
      if (_loginCode) {
        res = await fetchLoginByToken(_loginCode)
        if (res.data) {
          setToken(res.data.token)
          // 主动触发 storage 事件，确保其他组件能监听到变化
          window.dispatchEvent(
            new StorageEvent('storage', {
              key: '__TOKEN__',
              oldValue: token,
              newValue: res.data.token,
              url: window.location.href,
              storageArea: localStorage,
            }),
          )
          await getQuestionList()
          initConversation()
          dispatch(fetchConversations())
        }
      }
 else {
        await getQuestionList()
        initConversation()
        dispatch(fetchConversations())
      }
    }
 else {
      // 模拟登录 可以用来测试
      res = await fetchLoginByUid('123123')
      if (res.data) {
        setToken(res.data.token)
        // 主动触发 storage 事件，确保其他组件能监听到变化
        window.dispatchEvent(
          new StorageEvent('storage', {
            key: '__TOKEN__',
            oldValue: token,
            newValue: res.data.token,
            url: window.location.href,
            storageArea: localStorage,
          }),
        )
        await getQuestionList()
        initConversation()
        dispatch(fetchConversations())
      }
    }
  }, [setToken])

  // 修改 useEffect
  useEffect(() => {
    login()
  }, []) // 依赖数组为空，只在组件挂载时执行一次

  return (
    <div className={styles.homePage}>
      {isLoading && (
        <div className="w-full h-full flex justify-center items-center">
          <SdreamLoading />
        </div>
      )}
      <div className="h-full w-full">
        <div className="box flex flex-col h-full w-full">
          <div className="flex-1 items-center pt-[24px] sm:pt-[32px] scrollbar-hide">
            <div className="w-full">
              <div className="flex justify-center gap-[20px]">
                {/* 左侧区域 - 产品问答和您可以试着问我 */}
                  <div
                    className="flex flex-col gap-[20px] items-center overflow-y-auto scrollbar-hide"
                    style={{ height: isToolBtnActive ? 'calc(-64px + 100vh)' : '500px', background: isToolBtnActive ? 'linear-gradient(180deg, #F0F8FF 0%, #FFFFFF 50%, #FFFFFF 100%)' : '', borderRadius: '24px' }}
                  >
                {!isToolBtnActive && (
                  <motion.div className="w-full sm:w-auto" {...getAnimationProps(2)}>
                    <QuestionList
                      questions={productQuestions.content}
                      dotColor="#D4CCFF"
                      background="linear-gradient(180deg, #EBE6FF 0%, #FFFFFF 50%, #FFFFFF 100%)"
                      title={productQuestions.configName}
                      iconImg={HomeIcon1}
                      isToolBtn={isToolBtnActive}
                      isLoaded={isDataLoaded}
                    />
                  </motion.div>
                )}
                  <motion.div className="w-full sm:w-auto" {...getAnimationProps(3)}>
                    <QuestionList
                      questions={otherQuestions.content}
                      dotColor="#CBECFF"
                      background="linear-gradient(180deg, #F0F8FF 0%, #FFFFFF 50%, #FFFFFF 100%)"
                      height={isToolBtnActive ? '288px' : 'auto'}
                      title={otherQuestions.configName}
                      iconImg={HomeIcon2}
                      isToolBtn={isToolBtnActive}
                      isLoaded={isDataLoaded}
                    />
                  </motion.div>
                  {isToolBtnActive && (
                    <div>
                      <img src={SmartIce} alt="Smart Ice" className="w-[260px] h-[218px] mt-[-12px] object-contain" />
                    </div>
                  )}
                  </div>
                {/* 右侧区域 */}
                <div className="hidden sm:flex flex-1 h-full">
                  <div
                    className="w-full h-full bg-transparent box-border rounded-[24px]"
                    style={{ height: 'calc(100vh - 64px)', background: '#FFFFFF', padding: '0 30px' }}
                  >
                    <Outlet />
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}
