import React, { useEffect, useRef, useState } from 'react'
import { AnimatePresence, motion } from 'framer-motion'
import { Button, Tooltip } from '@heroui/react'
import { useLocalStorageState, useToggle } from 'ahooks'
import { useLocation, useNavigate, useSearchParams } from 'react-router-dom'
import { LoginModal } from '../LoginModal'
import type { RootState } from '@/store'
import SendIcon from '@/assets/svg/send.svg?react'
import { type WithAuthProps, withAuth } from '@/auth/withAuth'
import { useAppDispatch, useAppSelector } from '@/store/hook'
import { clearCurrentToolId, setCurrentToolId } from '@/store/conversationSlice'
import { setIsAsking } from '@/store/chatSlice'
import { fetchToolList } from '@/api/home'
import { getUserRolesForApi, safeSessionStorageGetItem, safeSessionStorageRemoveItem, safeSessionStorageSetItem } from '@/lib/utils'
import { fetchSessionConversationId, fetchUserQaRecordPage } from '@/api/conversation'

interface ChatEditorProps {
  onChange?: (value: string) => void
  onFocus?: () => void
  onSubmit?: (value: string, toolId?: string) => void
  onToolClick?: (isToolBtn: boolean, toolId?: string, toolName?: string, shouldChangeStyle?: boolean, conversationId?: string, isLoading?: boolean) => void
  placeholders: string[]
  showContentTips?: boolean
  initialValue?: string
  hideTools?: boolean // 隐藏工具选择部分
}
// onToolClick
const ChatEditorBase: React.FC<ChatEditorProps & WithAuthProps> = ({ checkAuth, onChange, onFocus, onSubmit, onToolClick, placeholders, showContentTips = false, initialValue = '', hideTools = false }) => {
  // const dispatch = useAppDispatch()
  const [content, setContent] = useState(initialValue)
  const dispatch = useAppDispatch()
  const editorRef = useRef<HTMLDivElement>(null)
  const [currentPlaceholder, setCurrentPlaceholder] = useState(0)
  const intervalRef = useRef<NodeJS.Timeout | null>(null)
  const [isOpenLoginModal, isOpenLoginModalActions] = useToggle()
  const isAsking = useAppSelector((state: RootState) => state.chat.isAsking)
  const [toolList, setToolList] = useState<any[]>([])
  const [selectedToolId, setSelectedToolId] = useState<string | null>(null)
  const [isToolBtnActive, setIsToolBtnActive] = useState<boolean>(true)
  const currentToolId = useAppSelector((state: RootState) => state.conversation.currentToolId)
  const [showToolQuestion, setShowToolQuestion] = useState<boolean>(false)
  const [sessionToolId, setSessionToolId] = useState<string | null>(null)
  const [searchParams, setSearchParams] = useSearchParams()
  const location = useLocation()
  const navigate = useNavigate()
  const toolIdFromUrl = searchParams.get('toolId')
  const fromCollect = location.state?.fromCollect

  /** 等待 token 就绪，确保登录接口返回后才继续 */
  const waitForToken = async (maxWaitTime = 3000): Promise<boolean> => {
    const startTime = Date.now()
    let initialToken = ''
    // 先获取初始 token 值
    try {
      const initialTokenStr = window.localStorage.getItem('__TOKEN__') || '""'
      initialToken = JSON.parse(initialTokenStr) || ''
    }
    catch {
      initialToken = ''
    }

    // 如果初始没有 token，等待 token 出现（从无到有，说明登录完成）
    if (!initialToken) {
      while (Date.now() - startTime < maxWaitTime) {
        const tokenStr = window.localStorage.getItem('__TOKEN__') || '""'
        let token = ''
        try {
          token = JSON.parse(tokenStr) || ''
        }
        catch {
          token = ''
        }
        if (token) {
          // token 从无到有，说明登录完成，再等待一小段时间确保写入完成
          await new Promise(resolve => setTimeout(resolve, 200))
          return true
        }
        // 等待 100ms 后再次检查
        await new Promise(resolve => setTimeout(resolve, 100))
      }
      return false
    }

    // 如果初始有 token，可能是旧 token 或新 token
    // 等待一小段时间，确保登录流程有足够时间完成（登录接口可能会更新 token）
    // 同时监听 token 变化
    let tokenChanged = false
    const checkTokenChange = () => {
      try {
        const currentTokenStr = window.localStorage.getItem('__TOKEN__') || '""'
        const currentToken = JSON.parse(currentTokenStr) || ''
        if (currentToken && currentToken !== initialToken) {
          tokenChanged = true
        }
      }
      catch {
        // 忽略解析错误
      }
    }

    // 监听 storage 事件（跨标签页）
    window.addEventListener('storage', checkTokenChange)
    // 轮询检查（同标签页）
    const interval = setInterval(checkTokenChange, 100)

    // 等待最多 1 秒，检测 token 是否变化
    const waitForChange = new Promise<void>((resolve) => {
      setTimeout(() => {
        window.removeEventListener('storage', checkTokenChange)
        clearInterval(interval)
        resolve()
      }, 1000)
    })
    await waitForChange

    // 如果 token 变化了，说明登录完成，再等待一小段时间确保写入完成
    if (tokenChanged) {
      await new Promise(resolve => setTimeout(resolve, 200))
      return true
    }

    // 如果 token 没有变化，可能是已经登录完成，但为了确保登录流程完成，再等待一小段时间
    await new Promise(resolve => setTimeout(resolve, 300))
    return true
  }

  // 获取工具列表
  const getToolList = async () => {
    try {
      // 等待 token 就绪后再调用接口
      await waitForToken()
      // 从路由中获取 userRoles 参数
      const userRoles = getUserRolesForApi()
      // 检查缓存（包含 userRoles 信息，有效期 5 分钟）
      const cacheKey = `toolList_${JSON.stringify(userRoles)}`
      const cached = sessionStorage.getItem(cacheKey)
      if (cached) {
        try {
          const { toolList, timestamp } = JSON.parse(cached)
          if (Date.now() - timestamp < 5 * 60 * 1000) {
            setToolList(toolList)
            return
          }
        }
        catch {
          // 缓存解析失败，继续调用接口
        }
      }
      // 调用真实 API 获取工具列表
      const res = await fetchToolList({ userRoles })
      if (res?.data && Array.isArray(res.data) && res.data.length > 0) {
        setToolList(res.data)
        // 缓存工具列表（包含 userRoles 和时间戳）
        sessionStorage.setItem(cacheKey, JSON.stringify({
          toolList: res.data,
          timestamp: Date.now(),
        }))
      }
    }
    catch (error) {
      console.error('获取工具列表失败:', error)
    }
  }

  // 根据 currentToolId 以及 sessionStorage 中的记录决定高亮逻辑
  useEffect(() => {
    if (currentToolId && !sessionToolId && !fromCollect) {
      // 清除过期的 Redux 值
      dispatch(clearCurrentToolId())
      // 如果 URL 中还有 toolId，也清除它
      if (toolIdFromUrl) {
        const newSearchParams = new URLSearchParams(searchParams)
        newSearchParams.delete('toolId')
        setSearchParams(newSearchParams, { replace: true })
      }
      setSelectedToolId(null)
      setIsToolBtnActive(true)
      return
    }

    if (!currentToolId && !sessionToolId && toolIdFromUrl) {
      const newSearchParams = new URLSearchParams(searchParams)
      newSearchParams.delete('toolId')
      setSearchParams(newSearchParams, { replace: true })
      setSelectedToolId(null)
      setIsToolBtnActive(true)
      return
    }

    // 只有当 sessionStorage 或 URL 中还能找到 toolId 时，才认为 Redux 中的 currentToolId 仍然有效
    const hasPersistentToolSource = Boolean(sessionToolId || toolIdFromUrl)

    if (currentToolId && hasPersistentToolSource) {
      setSelectedToolId(currentToolId)
      setIsToolBtnActive(false)
      return
    }

    if (sessionToolId) {
      setSelectedToolId(sessionToolId)
      setIsToolBtnActive(false)
      return
    }

    // 如果 currentToolId 和 sessionToolId 都没有值，根据路由中的 toolId 来决定
    if (toolIdFromUrl) {
      setSelectedToolId(toolIdFromUrl)
      setIsToolBtnActive(false)
    }
    else {
      setSelectedToolId(null)
      setIsToolBtnActive(true)
    }
  }, [currentToolId, sessionToolId, toolIdFromUrl, searchParams, setSearchParams, fromCollect])

  // 监听 sessionStorage 中的 currentToolId（历史点击时写入）来辅助高亮逻辑
  useEffect(() => {
    const syncSessionToolId = () => {
      const storedToolId = safeSessionStorageGetItem('currentToolId')
      // 如果 currentToolId 是空字符串，视为 null，确保制度活化能正确高亮
      setSessionToolId(storedToolId && storedToolId.trim() ? storedToolId : null)
    }
    syncSessionToolId()
    window.addEventListener('storage', syncSessionToolId)

    // 监听强制重置为制度活化的事件（登录成功后触发）
    const handleForceReset = () => {
      // 强制同步 sessionStorage（此时应该已经被清除了）
      syncSessionToolId()
      // 确保 Redux 也被清除（如果还没有）
      if (currentToolId) {
        dispatch(clearCurrentToolId())
      }
      // 强制设置为制度活化
      setSelectedToolId(null)
      setIsToolBtnActive(true)
    }
    window.addEventListener('forceResetToGeneralMode', handleForceReset)

    return () => {
      window.removeEventListener('storage', syncSessionToolId)
      window.removeEventListener('forceResetToGeneralMode', handleForceReset)
    }
  }, [currentToolId, dispatch])

  // 当路由变化时，同步更新 sessionToolId（因为 storage 事件不会在同标签页触发）
  useEffect(() => {
    const storedToolId = safeSessionStorageGetItem('currentToolId')
    // 如果 currentToolId 是空字符串，视为 null，确保制度活化能正确高亮
    setSessionToolId(storedToolId && storedToolId.trim() ? storedToolId : null)
    // 当路由切换（如点击历史记录）时，同步最新的 sessionStorage，避免同标签页删除后状态不同步
  }, [toolIdFromUrl, location.pathname])

  const startAnimation = () => {
    intervalRef.current = setInterval(() => {
      setCurrentPlaceholder(prev => (prev + 1) % placeholders.length)
    }, 3000)
  }
  const handleVisibilityChange = () => {
    if (document.visibilityState !== 'visible' && intervalRef.current) {
      clearInterval(intervalRef.current) // Clear the interval when the tab is not visible
      intervalRef.current = null
    }
    else if (document.visibilityState === 'visible') {
      startAnimation() // Restart the interval when the tab becomes visible
    }
  }

  const handleInput = () => {
    if (editorRef.current) {
      const newContent = editorRef.current.textContent || ''
      setContent(newContent)
      onChange?.(newContent)
    }
  }

  const handleSubmit = () => {
    if (isAsking)
      return
    if (checkAuth()) {
      if (content.trim()) {
        // 只在提质增效模式下传递 toolId，制度活化不传
        let toolId: string | undefined

        if (selectedToolId) {
          // 提质增效模式：使用选中的 toolId
          toolId = selectedToolId
        }
        else {
          // 制度活化：不传递 toolId
          toolId = undefined
        }
        onSubmit?.(content.trim(), toolId)
        setContent('')
        if (editorRef.current) {
          editorRef.current.textContent = ''
        }
      }
    }
  }

  const handleKeyDown = (e: React.KeyboardEvent) => {
    if (e.key === 'Enter' && !e.shiftKey) {
      e.preventDefault()
      handleSubmit()
    }
    else if (e.key === 'Backspace' && !content) {
      e.preventDefault() // 防止删除最后一个字符后继续删除
    }
  }
  const handleCloseLoginModal = () => {
    isOpenLoginModalActions.setLeft()
  }
  const handlePaste = (event: React.ClipboardEvent<HTMLDivElement>) => {
    event.preventDefault()
    const text = event.clipboardData.getData('text/plain')
    document.execCommand('insertText', false, text)
  }

  // 处理制度活化按钮点击：先创建新会话
  const handleGeneralClick = async () => {
    if (!checkAuth())
      return
    // 通知上层开始 loading
    onToolClick?.(true, undefined, '制度活化', false, undefined, true)
    try {
      // 重置提交按钮状态
      dispatch(setIsAsking(false))
      // 先更新 Redux，确保状态同步
      dispatch(clearCurrentToolId())
      // 立即更新本地状态，让 UI 立即响应
      setIsToolBtnActive(true)
      setSelectedToolId(null)
      safeSessionStorageRemoveItem('showToolQuestion')
      safeSessionStorageRemoveItem('currentToolId')
      setSessionToolId(null)
      setShowToolQuestion(false)
      // 先通知上层更新欢迎语（即便后续接口异常也能生效）
      onToolClick?.(true, undefined, '制度活化', false)
      // 清空路由中的 toolId 参数
      if (toolIdFromUrl) {
        const newSearchParams = new URLSearchParams(searchParams)
        newSearchParams.delete('toolId')
        setSearchParams(newSearchParams, { replace: true })
      }

      // 先调用 fetchSessionConversationId 获取会话ID
      // 等待 token 就绪后再调用接口
      await waitForToken()
      const requestData = {
        busiId: '',
      }
      const res = await fetchSessionConversationId(requestData)
      if (res?.data?.conversationId) {
        const conversationId = res.data.conversationId
        // 在 navigate 之前设置标记，避免 Chat 组件的 useEffect 重复调用接口
        sessionStorage.setItem('toolHistoryLoading', conversationId)
        // 更新路由到新的会话ID
        navigate(`/chat/${conversationId}`, {
          replace: true,
          state: {
            toolId: null,
            skipHistoryLoad: true, // 标记跳过历史记录加载，因为 ChatEditor 会处理
          },
        })
        // 使用获取到的会话ID调用历史会话
        await waitForToken()
        const qaRes = await fetchUserQaRecordPage(conversationId, '')
        console.log('qaRes chatEditor11111', qaRes)
        // 通过自定义事件将历史记录数据传递给父组件进行渲染
        window.dispatchEvent(new CustomEvent('toolHistoryLoaded', {
          detail: {
            conversationId,
            toolId: '',
            toolName: '制度活化',
            qaRecords: qaRes?.data || [],
          },
        }))
        // 清除标记，避免影响后续路由切换
        sessionStorage.removeItem('toolHistoryLoading')
      }
    }
    catch (error) {
      console.error('获取会话ID或历史记录失败:', error)
      // 出错时也要清除标记
      sessionStorage.removeItem('toolHistoryLoading')
    }
    finally {
      // 通知上层结束 loading
      onToolClick?.(true, undefined, '制度活化', false, undefined, false)
    }
  }

  // 处理工具按钮点击：先创建新会话，再切换工具
  const handleToolClick = async (tool: any) => {
    // 通知上层开始 loading
    onToolClick?.(false, tool.toolId, tool.toolName, true, undefined, true)
    try {
      // 重置提交按钮状态
      dispatch(setIsAsking(false))
      if (tool.toolName === '数据助手') {
        safeSessionStorageSetItem('showToolQuestion', 'true')
        setShowToolQuestion(true)
      }
      else {
        safeSessionStorageRemoveItem('showToolQuestion')
        setShowToolQuestion(false)
      }
      dispatch(setCurrentToolId(tool.toolId))
      setSelectedToolId(tool.toolId)
      setIsToolBtnActive(false)
      safeSessionStorageSetItem('currentToolId', tool.toolId)
      setSessionToolId(tool.toolId)

      // 先调用 fetchSessionConversationId 获取会话ID
      // 等待 token 就绪后再调用接口
      await waitForToken()
      const requestData = {
        busiId: tool.toolId,
      }
      const res = await fetchSessionConversationId(requestData)
      if (res?.data?.conversationId) {
        const conversationId = res.data.conversationId
        // 获取到会话ID后，通知上层更新欢迎语并传递 conversationId
        onToolClick?.(false, tool.toolId, tool.toolName, true, conversationId)
        // 在 navigate 之前设置标记，避免 Chat 组件的 useEffect 重复调用接口
        // 使用 sessionStorage 作为标记，因为 location.state 可能有时序问题
        sessionStorage.setItem('toolHistoryLoading', conversationId)
        // 更新路由到新的会话ID，并携带 toolId 参数
        navigate(`/chat/${conversationId}?toolId=${tool.toolId}`, {
          replace: true,
          state: {
            toolId: tool.toolId,
            skipHistoryLoad: true, // 标记跳过历史记录加载，因为 ChatEditor 会处理
          },
        })
        // 使用获取到的会话ID调用历史会话
        await waitForToken()
        const qaRes = await fetchUserQaRecordPage(conversationId, tool.toolId)
        console.log('qaRes chatEditor11111', qaRes)
        // 通过自定义事件将历史记录数据传递给父组件进行渲染
        window.dispatchEvent(new CustomEvent('toolHistoryLoaded', {
          detail: {
            conversationId,
            toolId: tool.toolId,
            toolName: tool.toolName,
            qaRecords: qaRes?.data || [],
          },
        }))
        // 清除标记，避免影响后续路由切换
        sessionStorage.removeItem('toolHistoryLoading')
      }
    }
    catch (error) {
      console.error('获取会话ID或历史记录失败:', error)
      // 出错时也要通知上层更新欢迎语（不传递 conversationId）
      onToolClick?.(false, tool.toolId, tool.toolName, true)
      // 出错时也要清除标记
      sessionStorage.removeItem('toolHistoryLoading')
    }
    finally {
      // 通知上层结束 loading
      onToolClick?.(false, tool.toolId, tool.toolName, true, undefined, false)
    }
  }

  useEffect(() => {
    startAnimation()
    document.addEventListener('visibilitychange', handleVisibilityChange)
    if (editorRef.current) {
      editorRef.current.style.height = 'auto'
      editorRef.current.style.height = `${editorRef.current.scrollHeight}px`
    }
    return () => {
      if (intervalRef.current) {
        clearInterval(intervalRef.current)
      }
      document.removeEventListener('visibilitychange', handleVisibilityChange)
    }
  }, [content])

  // 组件加载时和路由参数变化时获取工具列表
  useEffect(() => {
    if (!hideTools) {
      getToolList()
    }
  }, [location.pathname, location.search, hideTools])

  // 监听 sessionStorage 中的 showToolQuestion
  useEffect(() => {
    const checkShowToolQuestion = () => {
      const value = safeSessionStorageGetItem('showToolQuestion')
      setShowToolQuestion(value === 'true')
    }
    checkShowToolQuestion()
    // 监听 storage 事件以响应其他标签页的变化
    window.addEventListener('storage', checkShowToolQuestion)
    return () => {
      window.removeEventListener('storage', checkShowToolQuestion)
    }
  }, [])

  // 处理initialValue的变化
  useEffect(() => {
    if (initialValue && editorRef.current) {
      setContent(initialValue)
      editorRef.current.textContent = initialValue
      // 触发高度调整
      editorRef.current.style.height = 'auto'
      editorRef.current.style.height = `${editorRef.current.scrollHeight}px`
    }
  }, [initialValue])

  const [token] = useLocalStorageState<string | undefined>(
    '__TOKEN__',
    {
      defaultValue: '',
      listenStorageChange: true,
    },
  )

  return (
    <div
      className="flex items-end w-full h-auto relative mx-auto rounded-[24px] overflow-hidden transition duration-200 py-[8px] pl-[12px] pr-[12px] sm:py-[12px] sm:pl-[32px] sm:pr-[20px]"
      style={{
      background: '#FFFFFF',
      border: '1px solid #0a0a0a78',
      boxShadow: '0 8px 12px 0 rgba(235,235,235,0.30)',
      ...(!hideTools && toolList && toolList.length > 0 ? { height: '102px' } : {}),
    }}
    >
      <div
        ref={editorRef}
        contentEditable
        enterKeyHint="send"
        role="textbox"
        translate="no"
        className="w-full min-h-[40px] word-break-break-all max-h-[48px] p-2  rounded overflow-y-auto outline-none"
        onInput={handleInput}
        onFocus={onFocus}
        onKeyDown={handleKeyDown}
        onPaste={handlePaste}
        suppressContentEditableWarning
        data-placeholder={showToolQuestion ? '输入您想问的问题吧～' : '输入您想问的问题吧～'}
        style={{
          resize: 'none',
          maxHeight: '48px',
          wordBreak: 'break-all',
          ...(!hideTools && toolList && toolList.length > 0 ? { marginBottom: '40px' } : {}),
        }}
      >
      </div>
      <Tooltip isOpen={Boolean(token) && showContentTips && !isAsking && !content} color="foreground" content="请输入您的问题📖" placement="top-end">
        <Button className="ask-send" onPress={handleSubmit} radius="full" isDisabled={!content || isAsking} isIconOnly color="primary">
          <SendIcon />
        </Button>
      </Tooltip>
      <div className="absolute inset-0 flex items-center rounded-full pointer-events-none">
        <AnimatePresence mode="wait">
          {!content && (
            <motion.p
              initial={{
                y: 5,
                opacity: 0,
              }}
              key={`current-placeholder-${currentPlaceholder}`}
              animate={{
                y: 0,
                opacity: 1,
              }}
              exit={{
                y: -15,
                opacity: 0,
              }}
              transition={{
                duration: 0.3,
                ease: 'linear',
              }}
              className="dark:text-zinc-500 text-[14px] sm:text-[14px] font-normal text-[#3333334d] pl-4 sm:pl-12 text-left w-[calc(100%-2rem)] truncate"
            >
            </motion.p>
          )}
        </AnimatePresence>
      </div>
      {!hideTools && toolList && toolList.length > 0 && (
        <div className="absolute left-4 bottom-2 flex items-center gap-3 pointer-events-auto pl-[16px]">
          {toolList.map((tool: any, index: number) => {
            // 根据 selectedToolId 或路由中的 toolId 进行匹配，注意数据类型统一转换为字符串比较
            // 优先使用 selectedToolId（用户点击后的状态），其次使用 sessionStorage 中的 currentToolId（刷新时使用），再次使用路由中的 toolId（初始化时使用）
            const toolIdStr = String(tool.toolId)
            const isSelectedByState = selectedToolId && toolIdStr === String(selectedToolId)
            const isSelectedBySession = !selectedToolId && sessionToolId && toolIdStr === String(sessionToolId)
            // 制度活化高亮：路由内没有 toolId 或 toolId 为空时默认高亮，点击后也要高亮
            const isGeneralMode = tool.toolName === '制度活化' && isToolBtnActive && !selectedToolId && !sessionToolId && !toolIdFromUrl
            const isSelected = isSelectedByState || isSelectedBySession || isGeneralMode

            const baseBtnClass
              = 'w-auto h-[32px] px-3 rounded-full shadow-none text-[12px] flex items-center gap-2 transition-all duration-200 border'
            const selectedClass = isSelected
              ? ' bg-[#F3F7FF] border-[#AECBFF] text-[#165DFF]'
              : ' bg-[#FFFFFF] border-[#E6E8EB] text-[#111827]'
            const selectedVariant = isSelected ? 'solid' : 'bordered'
            const selectedColor = isSelected ? 'primary' : 'default'

            const handleButtonPress = async () => {
              // 高亮状态直接返回，避免重复触发
              if (isSelected)
                return
              if (tool.toolName === '制度活化')
                await handleGeneralClick()
              else
                await handleToolClick(tool)
            }

            // 处理后端返回的 base64 图标
            const getToolIconSrc = () => {
              const icon = tool.toolIcon
              if (!icon)
                return ''
              // 已经是完整的 data URL 或 http 链接时直接返回
              if (icon.startsWith('data:') || icon.startsWith('http'))
                return icon
              // 否则拼接为 base64 图片格式
              return `data:image/png;base64,${icon}`
            }

            return (
              <Button
                key={tool.toolId || `tool-${index}`}
                className={`${baseBtnClass}${selectedClass}`}
                radius="full"
                variant={selectedVariant}
                color={selectedColor}
                onPress={handleButtonPress}
              >
                {tool.toolIcon && (
                  <img
                    src={getToolIconSrc()}
                    className="w-4 h-4 flex-shrink-0"
                  />
                )}
                {tool.toolName}
              </Button>
            )
          })}
        </div>
      )}
      <LoginModal onClose={handleCloseLoginModal} isOpen={isOpenLoginModal} />
    </div>
  )
}

export const ChatEditor = withAuth(ChatEditorBase)
