import React, { useEffect, useRef, useState } from 'react'
import { AnimatePresence, motion } from 'framer-motion'
import { Button, Tooltip } from '@heroui/react'
import { useLocalStorageState, useToggle } from 'ahooks'
import { LoginModal } from '../LoginModal'
import type { RootState } from '@/store'
import SendIcon from '@/assets/svg/send.svg?react'
import { type WithAuthProps, withAuth } from '@/auth/withAuth'
import { useAppDispatch, useAppSelector } from '@/store/hook'
import { fetchToolList } from '@/api/home'
import { createConversation } from '@/store/conversationSlice'

function getUserRoleFromRoute() {
  try {
    const searchParams = new URLSearchParams(window.location.search)
    return searchParams.get('userRole') || undefined
  }
  catch {
    return undefined
  }
}

interface ChatEditorProps {
  onChange?: (value: string) => void
  onFocus?: () => void
  onSubmit?: (value: string, toolId?: string) => void
  onToolClick?: (isToolBtn: boolean, toolId?: string, shouldChangeStyle?: boolean) => void
  placeholders: string[]
  showContentTips?: boolean
  initialValue?: string
}
// onToolClick
const ChatEditorBase: React.FC<ChatEditorProps & WithAuthProps> = ({ checkAuth, onChange, onFocus, onSubmit, onToolClick, placeholders, showContentTips = false, initialValue = '' }) => {
  // const dispatch = useAppDispatch()
  const [content, setContent] = useState(initialValue)
  const dispatch = useAppDispatch()
  const editorRef = useRef<HTMLDivElement>(null)
  const [currentPlaceholder, setCurrentPlaceholder] = useState(0)
  const intervalRef = useRef<NodeJS.Timeout | null>(null)
  const [isOpenLoginModal, isOpenLoginModalActions] = useToggle()
  const isAsking = useAppSelector((state: RootState) => state.chat.isAsking)
  const [toolList, setToolList] = useState<any[]>([])
  const [selectedToolId, setSelectedToolId] = useState<string | null>(null)
  const [isToolBtnActive, setIsToolBtnActive] = useState<boolean>(true)
  const currentToolId = useAppSelector((state: RootState) => state.conversation.currentToolId)

  // 获取工具列表
  const getToolList = async () => {
    try {
      const userRole = getUserRoleFromRoute()
      const res = await fetchToolList(userRole ? { userRole } : undefined)
      if (res?.data) {
        // 根据 toolId 去重，防止重复渲染
        const uniqueList = res.data.filter((tool: any, index: number, self: any[]) =>
          index === self.findIndex((t: any) => t.toolId === tool.toolId),
        )
        setToolList(uniqueList)
      }
    }
    catch (error) {
      console.error('获取工具列表失败:', error)
    }
  }

  // 保持当前工具状态与 Redux 中的同步，确保跨页面返回时仍保持原模式
  useEffect(() => {
    if (currentToolId) {
      setSelectedToolId(currentToolId)
      setIsToolBtnActive(false)
    }
    else {
      setSelectedToolId(null)
      setIsToolBtnActive(true)
    }
  }, [currentToolId])

  const startAnimation = () => {
    intervalRef.current = setInterval(() => {
      setCurrentPlaceholder(prev => (prev + 1) % placeholders.length)
    }, 3000)
  }
  const handleVisibilityChange = () => {
    if (document.visibilityState !== 'visible' && intervalRef.current) {
      clearInterval(intervalRef.current) // Clear the interval when the tab is not visible
      intervalRef.current = null
    }
    else if (document.visibilityState === 'visible') {
      startAnimation() // Restart the interval when the tab becomes visible
    }
  }

  const handleInput = () => {
    if (editorRef.current) {
      const newContent = editorRef.current.textContent || ''
      setContent(newContent)
      onChange?.(newContent)
    }
  }

  const handleSubmit = () => {
    if (isAsking)
      return
    if (checkAuth()) {
      if (content.trim()) {
        // 只在提质增效模式下传递 toolId，通用模式不传
        let toolId: string | undefined

        if (!isToolBtnActive && selectedToolId) {
          // 提质增效模式：使用选中的 toolId
          toolId = selectedToolId
          // eslint-disable-next-line no-console
          console.log('📤 [ChatEditor] 提质增效模式提交，toolId:', toolId)
        }
        else {
          // 通用模式：不传递 toolId
          toolId = undefined
          // eslint-disable-next-line no-console
          console.log('📤 [ChatEditor] 通用模式提交，不传递 toolId')
        }
        onSubmit?.(content.trim(), toolId)
        setContent('')
        if (editorRef.current) {
          editorRef.current.textContent = ''
        }
      }
    }
  }

  const handleKeyDown = (e: React.KeyboardEvent) => {
    if (e.key === 'Enter' && !e.shiftKey) {
      e.preventDefault()
      handleSubmit()
    }
    else if (e.key === 'Backspace' && !content) {
      e.preventDefault() // 防止删除最后一个字符后继续删除
    }
  }
  const handleCloseLoginModal = () => {
    isOpenLoginModalActions.setLeft()
  }
  const handlePaste = (event: React.ClipboardEvent<HTMLDivElement>) => {
    event.preventDefault()
    const text = event.clipboardData.getData('text/plain')
    document.execCommand('insertText', false, text)
  }

  // 处理通用模式按钮点击：先创建新会话
  const handleGeneralClick = async () => {
    if (!checkAuth())
      return
    try {
      await dispatch(createConversation({
        conversationData: {},
        shouldNavigate: true,
        shouldSendQuestion: '',
      })).unwrap()
      setIsToolBtnActive(true)
      setSelectedToolId(null)
      onToolClick?.(true, undefined, false)
    }
    catch (error) {
      console.error('创建会话失败:', error)
    }
  }

  // 处理工具按钮点击：先创建新会话，再切换工具
  const handleToolClick = async (tool: any) => {
    if (!checkAuth())
      return
    try {
      await dispatch(createConversation({
        conversationData: {},
        shouldNavigate: true,
        shouldSendQuestion: '',
      })).unwrap()
      setSelectedToolId(tool.toolId)
      setIsToolBtnActive(false)
      onToolClick?.(false, tool.toolId, true)
    }
    catch (error) {
      console.error('创建会话失败:', error)
    }
  }

  useEffect(() => {
    startAnimation()
    document.addEventListener('visibilitychange', handleVisibilityChange)
    if (editorRef.current) {
      editorRef.current.style.height = 'auto'
      editorRef.current.style.height = `${editorRef.current.scrollHeight}px`
    }
    return () => {
      if (intervalRef.current) {
        clearInterval(intervalRef.current)
      }
      document.removeEventListener('visibilitychange', handleVisibilityChange)
    }
  }, [content])

  // 组件加载时获取工具列表
  useEffect(() => {
    getToolList()
  }, [])

  // 处理initialValue的变化
  useEffect(() => {
    if (initialValue && editorRef.current) {
      setContent(initialValue)
      editorRef.current.textContent = initialValue
      // 触发高度调整
      editorRef.current.style.height = 'auto'
      editorRef.current.style.height = `${editorRef.current.scrollHeight}px`
    }
  }, [initialValue])

  const [token] = useLocalStorageState<string | undefined>(
    '__TOKEN__',
    {
      defaultValue: '',
      listenStorageChange: true,
    },
  )

  return (
    <div
      className="flex items-end w-full h-auto relative mx-auto rounded-[24px] overflow-hidden transition duration-200 py-[8px] pl-[12px] pr-[12px] sm:py-[12px] sm:pl-[32px] sm:pr-[20px]"
      style={{
      background: '#FFFFFF',
      border: '1px solid #0a0a0a78',
      boxShadow: '0 8px 12px 0 rgba(235,235,235,0.30)',
      ...(toolList && toolList.length > 0 ? { height: '102px' } : {}),
    }}
    >
      <div
        ref={editorRef}
        contentEditable
        enterKeyHint="send"
        role="textbox"
        translate="no"
        className="w-full min-h-[40px] word-break-break-all max-h-[48px] p-2  rounded overflow-y-auto outline-none"
        onInput={handleInput}
        onFocus={onFocus}
        onKeyDown={handleKeyDown}
        onPaste={handlePaste}
        suppressContentEditableWarning
        data-placeholder="输入您想问的问题吧～"
        style={{
          resize: 'none',
          maxHeight: '48px',
          wordBreak: 'break-all',
          ...(toolList && toolList.length > 0 ? { marginBottom: '40px' } : {}),
        }}
      >
      </div>
      <Tooltip isOpen={Boolean(token) && showContentTips && !isAsking && !content} color="foreground" content="请输入您的问题📖" placement="top-end">
        <Button className="ask-send" onPress={handleSubmit} radius="full" isDisabled={!content || isAsking} isIconOnly color="primary">
          <SendIcon />
        </Button>
      </Tooltip>
      <div className="absolute inset-0 flex items-center rounded-full pointer-events-none">
        <AnimatePresence mode="wait">
          {!content && (
            <motion.p
              initial={{
                y: 5,
                opacity: 0,
              }}
              key={`current-placeholder-${currentPlaceholder}`}
              animate={{
                y: 0,
                opacity: 1,
              }}
              exit={{
                y: -15,
                opacity: 0,
              }}
              transition={{
                duration: 0.3,
                ease: 'linear',
              }}
              className="dark:text-zinc-500 text-[14px] sm:text-[14px] font-normal text-[#3333334d] pl-4 sm:pl-12 text-left w-[calc(100%-2rem)] truncate"
            >
            </motion.p>
          )}
        </AnimatePresence>
      </div>
      {toolList && toolList.length > 0 && (
        <div className="absolute left-4 bottom-2 flex items-center gap-3 pointer-events-auto pl-[16px]">
          {toolList.map((tool: any, index: number) => {
            // index === 1 的按钮（通用模式）在默认状态下高亮
            const isSelected = (selectedToolId === tool.toolId && !isToolBtnActive) || (index === 1 && isToolBtnActive)
            const handleButtonPress = async () => {
              // 高亮状态直接返回，避免重复触发
              if (isSelected)
                return
              if (index === 1)
                await handleGeneralClick()
              else
                await handleToolClick(tool)
            }
            return (
              <Button
                key={tool.toolId || `tool-${index}`}
                className={`w-auto h-[32px] px-3 rounded-full shadow-none text-[12px] flex items-center gap-2 transition-all duration-200 ${
                  isSelected
                    ? 'bg-[#F3F7FF] border-[#F3F7FF] text-[#165DFF]'
                    : 'bg-white border border-[#E6E8EB] text-[#111827]'
                }`}
                radius="full"
                variant="bordered"
                onPress={handleButtonPress}
              >
                {tool.toolIcon && (
                  <img
                    src={tool.toolIcon}
                    className="w-4 h-4 flex-shrink-0"
                  />
                )}
                {tool.toolName}
              </Button>
            )
          })}
        </div>
      )}
      <LoginModal onClose={handleCloseLoginModal} isOpen={isOpenLoginModal} />
    </div>
  )
}

export const ChatEditor = withAuth(ChatEditorBase)
