import type React from 'react'
import { motion } from 'framer-motion'
import { useEffect, useRef, useState } from 'react'
import { useNavigate } from 'react-router-dom'
import { useClickAway, useSessionStorageState } from 'ahooks'
import styles from './Navbar.module.less'
import { NavBarItem } from './components/NavBarItem'
import { clearNavigationFlag, createConversation } from '@/store/conversationSlice'
import type { WithAuthProps } from '@/auth/withAuth'
import { withAuth } from '@/auth/withAuth'
import { NAV_BAR_ITEMS } from '@/config/nav'
import { useAppDispatch, useAppSelector } from '@/store/hook'
import Logo from '@/assets/svg/logo.svg?react'
import { isMobile } from '@/utils'

interface NavbarProps {
  isHistoryVisible: boolean
  onSetHistoryVisible: (visible: boolean) => void
}

const NavbarBase: React.FC<NavbarProps & WithAuthProps> = ({ isHistoryVisible, checkAuth, onSetHistoryVisible }) => {
  const dispatch = useAppDispatch()
  const navigate = useNavigate()

  const { currentConversationId, shouldNavigateToNewConversation } = useAppSelector(state => state.conversation)

  const handleCreateConversation = () => {
    dispatch(createConversation({
      conversationData: {},
      shouldNavigate: true,
      shouldSendQuestion: '',
    }))
  }

  const [isH5NavVisible, setIsH5NavVisible] = useState(isMobile())

  const handleClick = (type: string | undefined) => {
    if (type === 'logo') {
      if (!isH5NavVisible) {
        navigate('/')
        setIsH5NavVisible(true)
      }
      else {
        setIsH5NavVisible(!isH5NavVisible)
      }
      return
    }
    setIsH5NavVisible(true)
    if (!checkAuth()) {
      return
    }

    if (type === 'add') {
      handleCreateConversation()
    }

    if (type === 'history') {
      onSetHistoryVisible(!isHistoryVisible)
    }

    if (type === 'collect') {
      navigate('/collect')
    }

    if (type === 'tools') {
      navigate('/tools')
    }
  }
  // const handleLogout = () => {
  //   onSetHistoryVisible(false)
  //   navigate('/')
  // }

  const [navBarVisibleLocal, setNavBarVisibleLocal] = useSessionStorageState<string | undefined>(
    '__NAV_BAR_VISIBLE_LOCAL__',
    {
      defaultValue: '0',
      listenStorageChange: true,
    },
  )

  const toggleNavBarVisible = () => {
    setNavBarVisibleLocal(navBarVisibleLocal === '1' ? '0' : '1')
  }

  useEffect(() => {
    if (shouldNavigateToNewConversation && currentConversationId) {
      navigate(`/chat/${currentConversationId}`)
      dispatch(clearNavigationFlag())
    }
  }, [shouldNavigateToNewConversation, currentConversationId, navigate, dispatch])

  const navRef = useRef<HTMLButtonElement>(null)
  useClickAway(() => {
    setIsH5NavVisible(true)
  }, navRef)

  return (
    <>
      <motion.nav
        ref={navRef}
        animate={navBarVisibleLocal === '0' ? 'hidden' : ''}
        className="h-full flex-shrink-0 flex flex-col items-center justify-center relative"
      >
        {/* hidden */}
        <motion.div className={`layoutNavBarJoyride ${styles.layoutNavBarAgent}  rounded-full ${isH5NavVisible ? 'h-[55px] bg-white/70 shadow-md' : 'h-[380px]'} w-[54px] overflow-hidden py-[8px] sm:w-[64px] sm:bg-white gap-[24px] sm:h-auto sm:rounded-3xl sm:flex sm:py-[24px] sm:px-[8px]`}>
          <NavBarItem isHistoryVisible={isHistoryVisible} onClick={handleClick} icon={Logo} label="" key="logo" type={isMobile() ? 'logo3' : 'logo'} />

          {NAV_BAR_ITEMS.map((item) => {
            return (
              <NavBarItem isHistoryVisible={isHistoryVisible} onClick={handleClick} icon={item.icon} label={item.label} key={item.key} type={item.key} />
            )
          })}
          {/* <User onLogout={handleLogout} /> */}
        </motion.div>
        <div onClick={toggleNavBarVisible} className={`${styles.sidebarAction} ${navBarVisibleLocal === '0' ? styles.open : ''}`}></div>
      </motion.nav>
      {/* <JoyrideStep /> */}
    </>
  )
}

export const Navbar = withAuth(NavbarBase)
