// 问答功能独立聊天页
import React, { useCallback, useEffect, useMemo, useRef, useState } from 'react'
import { useLocation, useNavigate, useParams, useSearchParams } from 'react-router-dom'
import { useLocalStorageState, useScroll } from 'ahooks'
import { Button } from '@heroui/react'
import { motion } from 'framer-motion'
import styles from '../Chat/Chat.module.less'
import { processApiResponse } from '../Chat/helper'
import { ChatItemUser } from '../Chat/components/ChatItem/ChatItemUser'
import { ChatAnswerBox } from '../Chat/components/ChatItem/ChatAnswerBox'
import { TacticsWelcome } from './components/TacticsWelcome'
import DeleteIcon from '@/assets/svg/delete.svg?react'
import RefreshIcon from '@/assets/svg/refresh.svg?react'
import { TacticsChatEditor } from '@/components/TacticsChatEditor'
import type { ChatRecord } from '@/types/chat'
import { fetchTacticsQaRecordPage } from '@/api/tactics'
import { fetchCheckTokenApi, fetchStreamResponse } from '@/api/chat'
import { fetchLoginByToken, fetchLoginByUid } from '@/api/common'
import { clearTacticsNavigationFlag, clearTacticsShouldSendQuestion, createTacticsConversation, deleteTacticsConversations, fetchTacticsConversations } from '@/store/tacticsSlice'
import type { RootState } from '@/store'
import { useAppDispatch, useAppSelector } from '@/store/hook'
import ScrollBtoIcon from '@/assets/svg/scrollBto.svg?react'
import { setIsAsking } from '@/store/chatSlice'
import SdreamLoading from '@/components/SdreamLoading'

export const TacticsChat: React.FC = () => {
  const { id } = useParams<{ id: string }>()
  const location = useLocation()
  const navigate = useNavigate()
  const viteOutputObj = import.meta.env.VITE_OUTPUT_OBJ || 'open'
  const [isLoading, setIsLoading] = useState(false)
  const [allItems, setAllItems] = useState<ChatRecord[]>([])
  const [historyDividerIndex, setHistoryDividerIndex] = useState<number | null>(null)
  const [historyDividerTime, setHistoryDividerTime] = useState<string | null>(null)
  const [hasCleared, setHasCleared] = useState(false)
  const [showClearConfirm, setShowClearConfirm] = useState(false)
  // 标记当前会话是否已有历史记录；null 表示尚未完成查询
  const [hasHistory, setHasHistory] = useState<boolean | null>(null)
  // 标记当前会话中最后一条 AI 回答的推荐问题是否仍在加载中（v1/query_recommend_question）
  const [isRecommendLoading, setIsRecommendLoading] = useState(false)
  const dispatch = useAppDispatch()
  const {
    shouldSendQuestion: shouldSendQuestionFromState,
    shouldNavigateToNewConversation,
    currentConversationId,
  } = useAppSelector((state: RootState) => state.tactics)
  const isAsking = useAppSelector((state: RootState) => state.chat.isAsking)
  const [searchParams] = useSearchParams()
  const hasFetched = useRef(false)
  // 使用 useLocalStorageState 管理 token，与原有逻辑保持一致
  const [token, setToken] = useLocalStorageState<string | undefined>('__TOKEN__', {
    defaultValue: '',
  })
  // 优先从 location.state 获取，其次从 Redux state 获取
  const shouldSendQuestion = (location.state as { shouldSendQuestion?: string } | null)?.shouldSendQuestion || shouldSendQuestionFromState
  const scrollableRef = useRef<HTMLDivElement | any>(null)
  const position = useScroll(scrollableRef)
  const currentIdRef = useRef<string | undefined>(id)
  const lastSentQuestionRef = useRef<string>('')
  const abortControllerRef = useRef<AbortController | null>(null)
  // 读取 from=tactics 且 version=2 的额外参数
  const tacticsMetaFromState = (location.state as { tacticsMeta?: any } | null)?.tacticsMeta
  const tacticsMetaFromQuery = useMemo(() => {
    const from = searchParams.get('from')
    const version = searchParams.get('version')
    if (from !== 'tactics' || version !== '2') {
      return undefined
    }
    return {
      version,
      taskId: searchParams.get('taskId') || undefined,
      pinBeginTime: searchParams.get('pinBeginTime') || undefined,
      pinEndTime: searchParams.get('pinEndTime') || undefined,
      partOrAll: searchParams.get('partOrAll') || undefined,
      channel: searchParams.get('channel') || undefined,
      channelName: searchParams.get('channelName') || undefined,
    }
  }, [searchParams])
  const tacticsMetaFromStorage = useMemo(() => {
    const raw = sessionStorage.getItem('tacticsMeta')
    if (!raw) {
      return undefined
    }
    try {
      const parsed = JSON.parse(raw)
      if (parsed?.version === '2') {
        return parsed
      }
    }
    catch {
      // ignore
    }
    return undefined
  }, [])
  const tacticsMeta = useMemo(() => tacticsMetaFromState || tacticsMetaFromQuery || tacticsMetaFromStorage, [tacticsMetaFromState, tacticsMetaFromQuery, tacticsMetaFromStorage])

  // 读取 place=user 时的 userMeta 参数
  const userMetaFromState = (location.state as { userMeta?: any } | null)?.userMeta
  const userMetaFromQuery = useMemo(() => {
    const place = searchParams.get('place')
    if (place !== 'user') {
      return undefined
    }
    return {
      place,
      cstId: searchParams.get('cstId') || undefined,
      userId: searchParams.get('userId') || undefined,
      numberType: searchParams.get('numberType') || undefined,
    }
  }, [searchParams])
  const userMetaFromStorage = useMemo(() => {
    const raw = sessionStorage.getItem('userMeta')
    if (!raw) {
      return undefined
    }
    try {
      const parsed = JSON.parse(raw)
      if (parsed?.place === 'user') {
        return parsed
      }
    }
    catch {
      // ignore
    }
    return undefined
  }, [])
  const userMeta = useMemo(() => userMetaFromState || userMetaFromQuery || userMetaFromStorage, [userMetaFromState, userMetaFromQuery, userMetaFromStorage])

  // 仅用于创建会话的额外参数（create_conversation）
  const getConversationExtra = useCallback(() => {
    if (userMeta?.place === 'user') {
      return {
        busiType: '02',
        busiId: userMeta.userId,
      }
    }
    if (tacticsMeta) {
      return {
        busiType: '02',
        busiId: tacticsMeta.taskId,
      }
    }
    return {}
  }, [tacticsMeta, userMeta])

  // 根据 userMeta.numberType 与调用类型（自动/正常）计算请求参数用的 numberType（A03/A04...）
  const getNumberTypeWithUserMeta = useCallback(
    (defaultNumberType: string, isAuto: boolean): string => {
      const numberType = userMeta?.numberType
      if (!numberType)
        return defaultNumberType

      // numberType 仅在 3~6 时生效，其它保持默认
      if (isAuto) {
        // 自动调用
        switch (numberType) {
          case '3':
            return 'A03'
          case '4':
            return 'A05'
          case '5':
            return 'A07'
          case '6':
            return 'A09'
          default:
            return defaultNumberType
        }
      }
      // 正常问答
      switch (numberType) {
        case '3':
          return 'A04'
        case '4':
          return 'A06'
        case '5':
          return 'A08'
        case '6':
          return 'A10'
        default:
          return defaultNumberType
      }
    },
    [userMeta],
  )

  const formatCurrentTime = () => {
    const date = new Date()
    const pad = (value: number) => value.toString().padStart(2, '0')
    return `${date.getFullYear()}-${pad(date.getMonth() + 1)}-${pad(date.getDate())} ${pad(date.getHours())}:${pad(date.getMinutes())}:${pad(date.getSeconds())}`
  }

  // 登录逻辑（复用原有逻辑，与 TacticsHome.tsx 保持一致）
  const login = useCallback(async () => {
    if (hasFetched.current) {
      return
    }
    hasFetched.current = true
    const url = new URL(window.location.href)
    const searchParams = new URLSearchParams(url.search)
    const _loginCode = searchParams.get('loginCode')
    let res = {} as any
    if (viteOutputObj === 'inner') {
      if (_loginCode) {
        res = await fetchLoginByToken(_loginCode)
        if (res.data) {
          setToken(res.data.token)
          window.dispatchEvent(
            new StorageEvent('storage', {
              key: '__TOKEN__',
              oldValue: token,
              newValue: res.data.token,
              url: window.location.href,
              storageArea: localStorage,
            }),
          )
          dispatch(fetchTacticsConversations())
        }
      }
      else {
        dispatch(fetchTacticsConversations())
      }
    }
    else {
      res = await fetchLoginByUid('123123')
      if (res.data) {
        setToken(res.data.token)
        window.dispatchEvent(
          new StorageEvent('storage', {
            key: '__TOKEN__',
            oldValue: token,
            newValue: res.data.token,
            url: window.location.href,
            storageArea: localStorage,
          }),
        )
        dispatch(fetchTacticsConversations())
      }
    }
  }, [setToken, dispatch, token, viteOutputObj])

  /** 处理正常stream的数据 */
  const handleStreamMesageData = (msg: any, question: string) => {
    setAllItems((prevItems) => {
      const newItems = [...prevItems] // 创建数组的浅拷贝
      // 从后往前找到最后一个 AI 项
      let lastAiIndex = -1
      for (let i = newItems.length - 1; i >= 0; i--) {
        if (newItems[i].role === 'ai') {
          lastAiIndex = i
          break
        }
      }
      // eslint-disable-next-line no-console
      console.log('[TacticsChat] handleStreamMesageData', {
        lastAiIndex,
        itemsLength: newItems.length,
        msgData: msg.content?.data,
        prevItems: prevItems.map(item => ({ role: item.role, hasAnswer: !!item.answerList?.[0]?.answer })),
      })
      if (lastAiIndex >= 0) {
        // 确保 answer 字段存在，如果不存在则使用空字符串
        const newAnswer = msg.content?.data?.answer || ''
        // 创建最后一项的新对象，合并现有数据和新的 answer
        const existingAnswer = newItems[lastAiIndex].answerList?.[0]?.answer || ''
        const originalAnswer = existingAnswer + newAnswer
        // 移除所有括号及其内容
        let filteredAnswer = originalAnswer.replace(/\([^)]*\)/g, '').trim()
        // 去除 [参考文档《任意内容》 《任意内容》...] 格式的内容
        filteredAnswer = filteredAnswer.replace(/\[参考文档(?:《[^》]*》\s*)+\]/g, '').trim()

        newItems[lastAiIndex] = {
          ...newItems[lastAiIndex],
          question,
          answerList: [
            {
              ...msg.content.data,
              isShow: false,
              answer: filteredAnswer,
            },
          ],
        }
        // eslint-disable-next-line no-console
        console.log('[TacticsChat] handleStreamMesageData:updated', {
          filteredAnswer,
          answerList: newItems[lastAiIndex].answerList,
          updatedItem: newItems[lastAiIndex],
        })
      }
      else {
        // 如果没有找到 AI 项，创建一个新的 AI 项

        console.warn('[TacticsChat] handleStreamMesageData: no AI item found, creating new one')
        const newAnswer = msg.content?.data?.answer || ''
        let filteredAnswer = newAnswer.replace(/\([^)]*\)/g, '').trim()
        filteredAnswer = filteredAnswer.replace(/\[参考文档(?:《[^》]*》\s*)+\]/g, '').trim()
        newItems.push({
          role: 'ai',
          question,
          answerList: [
            {
              ...msg.content.data,
              isShow: false,
              answer: filteredAnswer,
            },
          ],
        } as ChatRecord)
      }
      return newItems
    })
  }

  /** 处理超过最大条数的数据 */
  const handleChatMaxCount = (msg: any, question: string) => {
    setAllItems((prevItems) => {
      const newItems = [...prevItems]
      // 从后往前找到最后一个 AI 项
      let lastAiIndex = -1
      for (let i = newItems.length - 1; i >= 0; i--) {
        if (newItems[i].role === 'ai') {
          lastAiIndex = i
          break
        }
      }
      if (lastAiIndex >= 0) {
        newItems[lastAiIndex] = {
          ...newItems[lastAiIndex],
          question,
          answerList: [
            {
              ...msg.content.data,
              isShow: false,
              isChatMaxCount: true,
              endAnswerFlag: true,
              answer: '已达上限',
            },
          ],
        }
      }
      return newItems
    })
    // 达到最大轮数时，没有打字效果，需要手动设置 isAsking 为 false
    dispatch(setIsAsking(false))
  }

  /** 提交问题 */
  const handleSubmitQuestion = async (
    question: string | undefined,
    productCode?: string,
    toolId?: string,
    extra?: { busiType?: string, recordType?: string, numberType?: string, includeQuestion?: boolean, includeTacticsMeta?: boolean, includeUserMeta?: boolean },
  ) => {
    // 优先读取缓存中的 toolId，再回退到传参
    const sessionToolId = sessionStorage.getItem('currentToolId') ?? undefined
    const resolvedToolId = toolId ?? sessionToolId ?? undefined
    // 停止之前的请求
    if (abortControllerRef.current) {
      abortControllerRef.current.abort()
    }

    const isNew = allItems.length <= 1
    dispatch(setIsAsking(true))

    // 检查token
    await fetchCheckTokenApi()
    // 一次性添加用户问题和空的AI回答；如果不需要问题（自动触发场景），只添加 AI 占位
    const shouldIncludeQuestion = extra?.includeQuestion !== false && !!question
    setAllItems(prevItems => [
      ...prevItems,
      ...(shouldIncludeQuestion
        ? [
            {
              role: 'user',
              question,
            } as ChatRecord,
          ]
        : []),
      {
        role: 'ai',
        answerList: [{ answer: '' }],
      } as ChatRecord,
    ])

    // 创建新的 AbortController
    abortControllerRef.current = new AbortController()

    let fetchUrl = `/conversation/api/conversation/mobile/v1/submit_question_stream`

    const viteOutputObj = import.meta.env.VITE_OUTPUT_OBJ || 'open'

    let proxy = ''
    if (viteOutputObj === 'open') {
      proxy = import.meta.env.MODE !== 'prod' ? '/api' : '/dev-sdream-api'
    }
    else {
      proxy = import.meta.env.MODE === 'dev' ? '/api' : '/dev-sdream-api'
    }

    fetchUrl = proxy + fetchUrl

    const shouldSendQuestion = extra?.includeQuestion !== false
    const shouldIncludeTacticsMeta = extra?.includeTacticsMeta !== false
    const shouldIncludeUserMeta = extra?.includeUserMeta !== false

    // 去除缓存中可能存在的 from 字段，避免作为参数传递
    const { from: _omitFrom, ...safeTacticsMeta } = tacticsMeta || {}

    const requestBody: Record<string, any> = {
      conversationId: currentIdRef.current,
      stream: true,
      productCode,
      toolId: resolvedToolId,
      ...(extra?.busiType ? { busiType: extra.busiType } : {}),
      ...(extra?.recordType ? { recordType: extra.recordType } : {}),
      ...(extra?.numberType ? { numberType: extra.numberType } : {}),
    }
    // 优先使用 userMeta，如果存在 userMeta 且需要包含，则传入 userMeta 的字段
    // 注意：这里显式剔除 userMeta 自带的 numberType 和 place，避免覆盖映射后的 numberType，并按需去掉 place
    if (userMeta && shouldIncludeUserMeta) {
      const { place: _omitPlace, ...safeUserMeta } = userMeta
      Object.assign(requestBody, safeUserMeta)
    }
    // 如果没有 userMeta 或不需要包含 userMeta，则使用 tacticsMeta
    else if (tacticsMeta && shouldIncludeTacticsMeta) {
      Object.assign(requestBody, safeTacticsMeta)
    }
    // from=tactics 场景下补充业务识别参数
    if (userMeta?.place === 'user') {
      requestBody.busiType ??= '02'
      requestBody.busiId ??= userMeta.userId
      // 当缓存内存在 place=user 时，始终使用缓存中的 numberType 作为接口参数
      if (userMeta.numberType) {
        requestBody.numberType = userMeta.numberType
      }
    }
    else if (tacticsMeta) {
      requestBody.busiType ??= '02'
      requestBody.busiId ??= tacticsMeta.taskId
    }
    // 如果 includeQuestion 为 false，不展示问题但接口参数中仍要传递 question
    if (shouldSendQuestion || (question && extra?.includeQuestion === false)) {
      requestBody.question = question ?? ''
    }
    fetchStreamResponse(
      fetchUrl,
      requestBody,
      (msg) => {
        // eslint-disable-next-line no-console
        console.log('[TacticsChat] handleSubmitQuestion:onMessage', msg)
        // 检查是否已被取消
        if (abortControllerRef.current?.signal.aborted) {
          return
        }

        // 处理错误
        if (msg?.type === 'ERROR') {
          // 出错时需要重置 isAsking 状态
          dispatch(setIsAsking(false))
          // 如果是 AbortError，不显示错误
          if (msg.content?.name === 'AbortError') {
            return
          }
          // 其他错误也可以在这里显示错误消息
          return
        }

        // 正常的stream数据
        if (msg?.type === 'DATA' && msg?.content?.code === '00000000') {
          // eslint-disable-next-line no-console
          console.log('流式[TacticsChat] handleSubmitQuestion:success', msg)
          // eslint-disable-next-line no-console
          console.log('[TacticsChat] handleSubmitQuestion:data', msg.content?.data)
          handleStreamMesageData(msg, question || '')
        }
        if (msg?.type === 'DATA' && msg?.content?.code === '01010005') {
          handleChatMaxCount(msg, question || '')
          return
        }
        if (msg.type === 'END') {
          if (isNew) {
            setTimeout(() => {
              dispatch(fetchTacticsConversations())
            }, 2000)
          }
        }
      },
      abortControllerRef.current.signal,
    )
  }

  /** 获取qa记录 */
  const getUserQaRecordPage = useCallback(async (conversationId: string) => {
    setIsLoading(true)
    try {
      const res = await fetchTacticsQaRecordPage(conversationId)
      const qaRecords = res.data || []
      const hasHistoryFlag = qaRecords.length > 0
      // 始终添加 system 角色作为欢迎语
      const messages = [{ role: 'system' } as ChatRecord, ...processApiResponse(qaRecords)]
      setHistoryDividerIndex(hasHistoryFlag ? messages.length : null)
      setHistoryDividerTime(hasHistoryFlag ? formatCurrentTime() : null)
      // 处理历史记录中的参考文档标记
      const processedMessages = messages.map((item) => {
        if (item.role === 'ai' && item.answerList?.[0]?.answer) {
          return {
            ...item,
            answerList: item.answerList.map(answerItem => ({
              ...answerItem,
              answer: answerItem.answer
                ?.replace(/\([^)]*\)/g, '')
                .replace(/\[参考文档(?:《[^》]*》\s*)+\]/g, '')
                .trim(),
            })),
          }
        }
        return item
      })
      // 使用函数式更新，避免覆盖正在进行的流式响应
      setAllItems((prevItems) => {
        // 如果当前有正在进行的流式响应（存在空的AI项），保留它们
        const hasPendingAi = prevItems.some(item => item.role === 'ai' && (!item.answerList?.[0]?.answer || item.answerList[0].answer === ''))
        if (hasPendingAi && processedMessages.length > 1) {
          // 如果历史记录不为空，合并历史记录和正在进行的响应
          const lastPendingAiIndex = prevItems.length - 1
          const lastPendingAi = prevItems[lastPendingAiIndex]
          if (lastPendingAi?.role === 'ai') {
            return [...processedMessages, lastPendingAi]
          }
        }
        return processedMessages
      })
      setHasCleared(false)
      setHasHistory(hasHistoryFlag)
    }
    catch {
      // 如果获取失败，至少显示欢迎语
      setAllItems((prevItems) => {
        const hasPendingAi = prevItems.some(item => item.role === 'ai' && (!item.answerList?.[0]?.answer || item.answerList[0].answer === ''))
        if (hasPendingAi) {
          const lastPendingAiIndex = prevItems.length - 1
          const lastPendingAi = prevItems[lastPendingAiIndex]
          if (lastPendingAi?.role === 'ai') {
            return [{ role: 'system' } as ChatRecord, lastPendingAi]
          }
        }
        return [{ role: 'system' } as ChatRecord]
      })
      setHasCleared(false)
      setHistoryDividerIndex(null)
      setHistoryDividerTime(null)
      // 拉取失败时视为无历史记录，后续可按需触发自动提问
      setHasHistory(false)
    }
    finally {
      setIsLoading(false)
    }
  }, [])

  /** 点击滚动到底部 */
  const scrollToBottom = () => {
    scrollableRef.current.scrollTo(scrollableRef.current.scrollHeight, { behavior: 'smooth' })
  }

  const handleClearRecord = useCallback(async () => {
    if (!currentIdRef.current)
      return
    setShowClearConfirm(false)
    try {
      await dispatch(deleteTacticsConversations([currentIdRef.current])).unwrap()
      // 停止正在进行的请求
      if (abortControllerRef.current) {
        abortControllerRef.current.abort()
        dispatch(setIsAsking(false))
      }
      // 清空对话列表，只保留欢迎语
      setAllItems([{ role: 'system' } as ChatRecord])
      setHasCleared(true)
      setHistoryDividerIndex(null)
      setHistoryDividerTime(null)
      dispatch(
        createTacticsConversation({
          conversationData: getConversationExtra(),
          shouldNavigate: false,
          shouldSendQuestion: '',
        }),
      )
    }
    catch (error) {
      console.error('清除记录失败:', error)
    }
  }, [dispatch, getConversationExtra])

  const handleReanalyze = useCallback(async () => {
    // 正在回答或尚未就绪时不触发重新分析
    if (!currentIdRef.current || isLoading || isAsking) {
      // eslint-disable-next-line no-console
      console.log('[TacticsChat] handleReanalyze blocked', {
        currentId: currentIdRef.current,
        isLoading,
        isAsking,
      })
      return
    }
    // 重新触发一次提问，与首次自动调用保持一致：
    // - 无 userMeta：沿用原逻辑，传 recordType=A02，使用 tacticsMeta
    // - 有 userMeta：按 userMeta.numberType 映射 numberType（A03/A05/A07/A09），不传 recordType，仅传 userMeta
    const hasUserMeta = !!userMeta
    if (hasUserMeta) {
      const mappedNumberType = getNumberTypeWithUserMeta('A02', true)
      await handleSubmitQuestion(undefined, undefined, undefined, {
        busiType: '02',
        recordType: mappedNumberType,
        includeQuestion: false,
        includeTacticsMeta: false,
        includeUserMeta: true,
      })
    }
    else {
      await handleSubmitQuestion(undefined, undefined, undefined, {
        busiType: '02',
        recordType: 'A01',
        includeQuestion: false,
        includeTacticsMeta: true,
      })
    }
    // 重新拉取会话列表，保持原有行为
    await dispatch(fetchTacticsConversations())
    if (currentIdRef.current) {
      navigate(`/tactics/chat/${currentIdRef.current}`)
    }
  }, [dispatch, navigate, handleSubmitQuestion, userMeta, getNumberTypeWithUserMeta, isLoading, isAsking])

  const handleCancelClear = useCallback(() => {
    setShowClearConfirm(false)
  }, [])

  const handleOpenClearConfirm = useCallback(() => {
    setShowClearConfirm(true)
  }, [])

  useEffect(() => {
    if (id) {
      // 停止之前的请求
      if (abortControllerRef.current) {
        abortControllerRef.current.abort()
        dispatch(setIsAsking(false))
      }

      currentIdRef.current = id
      lastSentQuestionRef.current = ''
      // 每次切换会话时重置历史标记，等待新会话的历史查询结果
      setHasHistory(null)
      getUserQaRecordPage(id)
    }
    else {
      // 如果没有 id，显示欢迎语，等待登录成功后创建新会话
      setAllItems([{ role: 'system' } as ChatRecord])
      setHistoryDividerIndex(null)
      setHistoryDividerTime(null)
      setIsLoading(false)
      setHasHistory(false)
    }
  }, [id, getUserQaRecordPage, dispatch])

  // 初始化时调用登录（登录成功后会自动创建会话）
  useEffect(() => {
    login()
  }, [login])

  // 进入会话后自动触发一次提交，默认 question 为“策略分析”，busiType 02 / recordType A02
  const hasAutoSubmittedRef = useRef(false)
  useEffect(() => {
    if (id) {
      // 当 id 变化时，重置自动提交标志
      hasAutoSubmittedRef.current = false
    }
  }, [id])
  useEffect(() => {
    // 仅在历史记录查询完成且为空时，才自动触发一次提交
    if (currentIdRef.current && !isLoading && hasHistory === false && !hasAutoSubmittedRef.current) {
      hasAutoSubmittedRef.current = true
      // 进入会话后自动触发一次提交：
      // - 无 userMeta：沿用原逻辑，recordType=A02，携带 tacticsMeta
      // - 有 userMeta：按 userMeta.numberType 映射 numberType（A03/A05/A07/A09），不传 recordType，仅传 userMeta
      const hasUserMeta = !!userMeta
      if (hasUserMeta) {
        const mappedNumberType = getNumberTypeWithUserMeta('A02', true)
        handleSubmitQuestion(
          undefined,
          undefined,
          undefined,
          {
            busiType: '02',
            recordType: mappedNumberType,
            includeQuestion: false,
            includeTacticsMeta: false,
            includeUserMeta: true,
          },
        )
      }
      else {
        handleSubmitQuestion(
          undefined,
          undefined,
          undefined,
          {
            busiType: '02',
            recordType: 'A01',
            includeQuestion: false,
            includeTacticsMeta: true,
          },
        )
      }
    }
  }, [isLoading, handleSubmitQuestion, id, userMeta, getNumberTypeWithUserMeta, hasHistory])

  // 创建新会话成功后跳转到新会话页面
  useEffect(() => {
    if (shouldNavigateToNewConversation && currentConversationId) {
      navigate(`/tactics/chat/${currentConversationId}`, {
        state: {
          shouldSendQuestion: shouldSendQuestionFromState,
          tacticsMeta,
          userMeta,
        },
        replace: true,
      })
      dispatch(clearTacticsNavigationFlag())
    }
  }, [shouldNavigateToNewConversation, currentConversationId, navigate, dispatch, shouldSendQuestionFromState, tacticsMeta, userMeta])

  // 处理shouldSendQuestion的变化 - 自动发送问题
  useEffect(() => {
    // eslint-disable-next-line no-console
    console.log('[TacticsChat] autoSend check', {
      shouldSendQuestion,
      currentId: currentIdRef.current,
      isLoading,
      lastSentQuestion: lastSentQuestionRef.current,
    })
    if (
      shouldSendQuestion
      && currentIdRef.current
      && !isLoading
      && shouldSendQuestion !== lastSentQuestionRef.current
    ) {
      lastSentQuestionRef.current = shouldSendQuestion
      dispatch(clearTacticsShouldSendQuestion())
      const hasUserMeta = !!userMeta
      setTimeout(() => {
        if (hasUserMeta) {
          const mappedNumberType = getNumberTypeWithUserMeta('A01', false)
          handleSubmitQuestion(
            shouldSendQuestion,
            undefined,
            undefined,
            {
              includeQuestion: true,
              includeTacticsMeta: false,
              includeUserMeta: true,
              recordType: mappedNumberType,
            },
          )
        }
        else {
          // 无 userMeta：保持原有行为，不传任何 extra（不携带 numberType），由内部默认逻辑处理 tacticsMeta
          handleSubmitQuestion(shouldSendQuestion)
        }
      }, 100)
    }
  }, [shouldSendQuestion, isLoading, dispatch, userMeta, handleSubmitQuestion, getNumberTypeWithUserMeta])

  // 提供给 ChatAnswerBox/推荐问题的提交方法，保持与手动提问一致的 recordType 逻辑
  const handleAnswerBoxSubmit = useCallback(
    (question: string, productCode?: string) => {
      const hasUserMeta = !!userMeta
      if (hasUserMeta) {
        const mappedNumberType = getNumberTypeWithUserMeta('A02', false)
        return handleSubmitQuestion(
          question,
          productCode,
          undefined,
          {
            busiType: '02',
            recordType: mappedNumberType,
            includeQuestion: true,
            includeTacticsMeta: false,
            includeUserMeta: true,
          },
        )
      }
      return handleSubmitQuestion(
        question,
        productCode,
        undefined,
        {
          busiType: '02',
          recordType: 'A02',
          includeQuestion: true,
          includeTacticsMeta: false,
        },
      )
    },
    [userMeta, getNumberTypeWithUserMeta, handleSubmitQuestion],
  )

  return (
    <div
      className={styles.scrollView}
      style={{
        height: '100vh',
        display: 'flex',
        flexDirection: 'column',
        marginLeft: 'auto',
        marginRight: 0,
        backgroundColor: '#FFFFFF',
        zIndex: 1000,
      }}
    >
      {/* 右上操作区 */}
      <div className="flex justify-end items-center gap-[16px] px-[16px] pt-[16px] pb-[8px]">
        <button
          type="button"
          onClick={handleOpenClearConfirm}
          className="flex items-center gap-[4px] text-[#4A90E2] text-[14px] hover:opacity-80 transition-opacity cursor-pointer bg-transparent border-none outline-none"
        >
          <DeleteIcon className="w-[16px] h-[16px] text-[#B2B8C1]" />
          <span>清除记录</span>
        </button>
        <button
          type="button"
          onClick={handleReanalyze}
          disabled={!currentIdRef.current || isLoading || isAsking || isRecommendLoading}
          className={`flex items-center gap-[4px] text-[14px] transition-opacity bg-transparent border-none outline-none ${
            !currentIdRef.current || isLoading || isAsking || isRecommendLoading
              ? 'text-[#B2B8C1] cursor-not-allowed opacity-60'
              : 'text-[#4A90E2] hover:opacity-80 cursor-pointer'
          }`}
        >
          <RefreshIcon className="w-[16px] h-[16px] text-[#B2B8C1]" />
          <span>重新分析</span>
        </button>
      </div>
      <div className={`${styles.chatPage} relative flex flex-col h-full`}>
        {showClearConfirm && (
          <div className="fixed inset-0 z-[2000] flex items-center justify-center bg-black/40 px-4">
            <div className="bg-white rounded-[12px] p-6 w-full max-w-[360px] shadow-lg">
              <div className="text-[16px] font-medium text-[#1F2937] mb-4">是否确定清空历史记录？</div>
              <div className="flex justify-end gap-3">
                <button
                  type="button"
                  onClick={handleCancelClear}
                  className="px-4 py-2 rounded-[8px] border border-[#E5E7EB] text-[#4B5563] hover:bg-[#F9FAFB]"
                >
                  取消
                </button>
                <button
                  type="button"
                  onClick={handleClearRecord}
                  className="px-4 py-2 rounded-[8px] bg-[#2563EB] text-white hover:opacity-90"
                >
                  确定
                </button>
              </div>
            </div>
          </div>
        )}
        <div className={`${styles.content} flex-1 overflow-hidden flex flex-col`}>
          {isLoading && (
            <div className="w-full h-full flex justify-center items-center">
              <SdreamLoading />
            </div>
          )}
          {!isLoading && (
            <motion.div
              ref={scrollableRef}
              initial={{ opacity: 0, y: -10 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{
                duration: 0.3,
                opacity: { duration: 0.1 },
              }}
              className={`${styles.scrollable} scrollbar-hide scroll-smooth flex-1 overflow-y-auto px-[16px] min-h-0`}
            >
              <div className={`${styles.inter} pt-[8px] pb-[24px] min-h-full`}>
                {allItems.map((record, index) => {
                  const recordId = record.answerList?.[0]?.recordId || record.groupId
                  const uniqueKey = recordId
                    ? `${record.role}-${recordId}`
                    : `${record.role}-${record.question || record.answerList?.[0]?.answer || ''}-${index}`
                  return (
                    <React.Fragment key={uniqueKey}>
                      {historyDividerIndex !== null && index === historyDividerIndex && (
                        <div className="flex items-center gap-[12px] my-[16px] text-[#B2B8C1] text-[12px]">
                          <div className="flex-1 h-px bg-[#E4E7EC]" />
                          <span>{historyDividerTime || formatCurrentTime()}</span>
                          <div className="flex-1 h-px bg-[#E4E7EC]" />
                        </div>
                      )}
                      <div
                        className="w-full chatItem mx-auto"
                      >
        {record.role === 'system' && <TacticsWelcome cleared={hasCleared} />}
                        {record.role === 'user' && <ChatItemUser record={record} />}
                        {record.role === 'ai' && (
                          <ChatAnswerBox
                            onSubmitQuestion={handleAnswerBoxSubmit}
                            isLastAnswer={index === allItems.length - 1}
                            showIndex={0}
                            record={record}
                            index={index}
                            // 仅对最后一条 AI 回答的推荐问题 loading 状态进行跟踪
                            onRecommendLoadingChange={index === allItems.length - 1 ? setIsRecommendLoading : undefined}
                          />
                        )}
                      </div>
                    </React.Fragment>
                  )
                })}
              </div>
            </motion.div>
          )}
        </div>
        <div className="relative box-border px-[16px] mx-auto iptContainer w-full flex-shrink-0 pb-[18px] pt-[12px] bg-white border-t border-gray-100">
          <div className="relative">
            <div className="absolute left-1/2 ml-[-20px] top-[-45px]">
              <motion.div
                initial="hidden"
                animate={(position?.top as number) < -20 ? 'visible' : 'hidden'}
                variants={{
                  hidden: { opacity: 0, y: 20, pointerEvents: 'none' as const },
                  visible: { opacity: 1, y: 0, pointerEvents: 'auto' as const },
                }}
                transition={{ duration: 0.3, ease: 'easeInOut' }}
              >
                <Button onPress={scrollToBottom} radius="full" isIconOnly color="primary">
                  <ScrollBtoIcon />
                </Button>
              </motion.div>
            </div>
            <TacticsChatEditor
              onSubmit={(question) => {
                // eslint-disable-next-line no-console
                console.log('[TacticsChat] onSubmit wrapper', {
                  question,
                  currentId: currentIdRef.current,
                  isLoading,
                })
                // 正常问答：
                // - 无 userMeta：保持原有逻辑，recordType=A01、busiType=02、不带 numberType
                // - 有 userMeta：按 userMeta.numberType 映射 numberType（A04/A06/A08/A10），不传 recordType，仅传 userMeta
                const hasUserMeta = !!userMeta
                if (hasUserMeta) {
                  const mappedNumberType = getNumberTypeWithUserMeta('A02', false)
                  return handleSubmitQuestion(
                    question,
                    undefined,
                    undefined,
                    {
                      busiType: '02',
                    recordType: mappedNumberType,
                      includeQuestion: true,
                      includeTacticsMeta: false,
                      includeUserMeta: true,
                    },
                  )
                }
                return handleSubmitQuestion(
                  question,
                  undefined,
                  undefined,
                  {
                    busiType: '02',
                    recordType: 'A02',
                    includeQuestion: true,
                    includeTacticsMeta: false,
                  },
                )
              }}
              placeholders={['']}
            />
            <div className="w-full text-center mt-[12px] text-[#3333334d] text-[12px]">
              内容由AI模型生成，其准确性和完整性无法保证，仅供参考
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}
