import { Avatar, Button } from '@heroui/react'
import { useEffect, useState } from 'react'
import { ChatAnswerShower } from './ChatAnswerShower'
import { ChatAnswerParser } from './ChatAnswerParser'
import { ChatAnswerRecommend } from './ChatAnswerRecommend'
import { ChatMaxCount } from './ChatMaxCount'
import type { Answer, ChatRecord } from '@/types/chat'
import AvatarBot from '@/assets/avatarBot.png'
import AIIcon from '@/assets/ai-icon.png'
import { useAppDispatch } from '@/store/hook'
import { setIsAsking } from '@/store/chatSlice'
import SdreamLoading from '@/components/SdreamLoading'

interface ChatAnswerBoxProps {
  record: ChatRecord
  showIndex: number
  isLastAnswer: boolean
  index: number
  onSubmitQuestion: (question: string, productCode?: string) => void
  onRecommendLoadingChange?: (loading: boolean) => void
}

export const ChatAnswerBox: React.FC<ChatAnswerBoxProps> = ({ record, showIndex, isLastAnswer, onSubmitQuestion, onRecommendLoadingChange }) => {
  const [isShowRecommend, setIsShowRecommend] = useState(false)
  const [recommendUseAnswer, setRecommendUseAnswer] = useState<Answer>()
  const [innerRecord, setInnerRecord] = useState<ChatRecord>(record)
  const [isTyping, setIsTyping] = useState(false)
  const dispatch = useAppDispatch()
  const viteOutputObj = import.meta.env.VITE_OUTPUT_OBJ || 'open'

  const handleTyping = () => {
    setIsTyping(true)
  }

  const handleComplate = (answer: Answer) => {
    setIsShowRecommend(true)
    setRecommendUseAnswer(answer)
    dispatch(setIsAsking(false))
    setIsTyping(false)
  }

  const handleStopTyping = () => {
    const _innerRecord = JSON.parse(JSON.stringify(innerRecord))
    _innerRecord.answerList[showIndex].isStopTyping = true
    setInnerRecord(_innerRecord)
  }

  useEffect(() => {
    setInnerRecord(record)
  }, [record])

  /**
   * 当当前 AI 记录没有对应的问题文本时（例如策略页进入会话后的自动分析、
   * 点击「重新分析」触发的自动问题），需要隐藏点赞 / 点踩 / 复制按钮。
   * 对于正常问答（有用户问题）则保持原有行为。
   */
  const hideOperateForRecord = !innerRecord.question

  return (
    <div>
      {innerRecord.answerList.map((item, index) => {
        return (
          index === showIndex && (
            <div className="chatItemBotContainer w-full" key={`${item.recordId}-${index}`}>
              <div className="flex">
                <Avatar
                  className="sm:mr-[20px] hidden sm:block flex-shrink-0"
                  src={viteOutputObj === 'inner' ? AIIcon : AvatarBot}
                />
                <div
                  style={{ background: '#F7FAFD' }}
                  className="rounded-[20px] box-border px-[16px] py-[12px] sm:px-[24px] sm:py-[20px] relative"
                >
                  {item.answer?.length || item.cardList?.length
? (
                    <div className="content">
                      {item.isShow && (
                        <ChatAnswerShower
                          onSubmitQuestion={onSubmitQuestion}
                          isLastAnswer={isLastAnswer}
                          answer={item}
                          hideOperate={hideOperateForRecord}
                        />
                      )}
                      {!item.isShow && !item.isChatMaxCount && (
                        <ChatAnswerParser
                          onSubmitQuestion={onSubmitQuestion}
                          isLastAnswer={isLastAnswer}
                          isStopTyping={item.isStopTyping}
                          onTyping={handleTyping}
                          onComplate={() => handleComplate(item)}
                          answer={item}
                          hideOperate={hideOperateForRecord}
                        />
                      )}
                      {!item.isShow && item.isChatMaxCount && <ChatMaxCount />}
                    </div>
                  )
: (
                    <SdreamLoading />
                  )}
                </div>
                <div className="hidden sm:block w-[65px] flex-shrink-0"></div>
              </div>
              {isTyping && (
                <div className="sm:pl-[62px] mt-[12px]">
                  <Button onClick={handleStopTyping} color="primary" variant="bordered">
                    停止生成
                  </Button>
                </div>
              )}

              {isLastAnswer && !item.isChatMaxCount && isShowRecommend && recommendUseAnswer && (
                <ChatAnswerRecommend
                  onSubmitQuestion={onSubmitQuestion}
                  answer={recommendUseAnswer}
                  onLoadingChange={onRecommendLoadingChange}
                />
              )}
              <div className="h-[20px] sm:h-[32px] w-full"></div>
            </div>
          )
        )
      })}
    </div>
  )
}
