import type { PayloadAction } from '@reduxjs/toolkit'
import { createAsyncThunk, createSlice } from '@reduxjs/toolkit'
import { fetchUserQaRecordPage } from '@/api/conversation'
import type { ChatRecord, ChatState, OriginalRecord } from '@/types/chat'

const initialState: ChatState = {
  records: [],
  isLoading: false,
  error: null,
}

function processApiResponse(data: OriginalRecord[]): ChatRecord[] {
  const chatRecord: ChatRecord[] = []
  if (data.length === 0)
    return chatRecord
  data.forEach((record) => {
    chatRecord.push({
      type: 'question',
      originalData: record,
    })

    if (record.answerList && record.answerList.length > 0) {
      chatRecord.push({
        type: 'answer',
        originalData: record,
      })
    }
  })

  return chatRecord
}

export const fetchChatRecords = createAsyncThunk(
  'chat/fetchRecords',
  async (conversationId: string) => {
    const response = await fetchUserQaRecordPage(conversationId)
    return processApiResponse(response.data)
  },
)

const chatSlice = createSlice({
  name: 'chat',
  initialState,
  reducers: {
    addRecord: (state, action: PayloadAction<ChatRecord>) => {
      state.records.push(action.payload)
    },
    updateLastAnswer: (state, action: PayloadAction<OriginalRecord>) => {
      const lastIndex = state.records.length - 1
      if (lastIndex >= 0 && state.records[lastIndex].type === 'streamAnswer') {
        state.records[lastIndex] = {
          ...state.records[lastIndex],
          originalData: {
            ...state.records[lastIndex].originalData,
            answerList: [
              {
                ...state.records[lastIndex].originalData.answerList[0],
                answer: state.records[lastIndex].originalData.answerList[0].answer + action.payload.answer,
              },
            ],
          },
        }
      }
    },
    setIsLoading: (state, action: PayloadAction<boolean>) => {
      state.isLoading = action.payload
    },
  },
  extraReducers: (builder) => {
    builder
      .addCase(fetchChatRecords.pending, (state) => {
        state.isLoading = true
        state.error = null
      })
      .addCase(fetchChatRecords.fulfilled, (state, action) => {
        state.isLoading = false
        state.records = action.payload
      })
      .addCase(fetchChatRecords.rejected, (state, action) => {
        state.isLoading = false
        state.error = action.error.message || 'Failed to fetch chat records'
      })
  },
})

export const { addRecord, setIsLoading, updateLastAnswer } = chatSlice.actions
export default chatSlice.reducer
