import { ArgumentMetadata, HttpException, HttpStatus, Injectable, PipeTransform, Type, } from '@nestjs/common';
import { plainToClass } from 'class-transformer';
import { ValidationError, validate } from 'class-validator';

function formatErrors(errors: ValidationError[], errorMessage?: any, parentField?: string) {
    const message = errorMessage || {}
    let errorField = ''
    let validationList
    errors.forEach(error => {
        errorField = parentField ? `${parentField}.${error.property}` : error?.property
        if(!error?.constraints && error?.children?.length) {
            formatErrors(error.children, message, errorField)
        } else {
            validationList = Object.values(error.constraints)
            message[errorField] = validationList.length > 0 ? validationList.pop() : 'Invalid Value'!
        }
    })
    console.log(message,'message')
    return message
}

@Injectable()
export class ValidationPipe implements PipeTransform<any> {
    async transform(value: any, metadata: ArgumentMetadata) {
        const { metatype } = metadata;
        if (!metatype || !this.toValidate(metatype)) return value;
        const object = plainToClass(metatype, value);
        const errors = await validate(object);
        if (errors.length > 0) {
            const errorObj = formatErrors(errors)
            throw new HttpException(
                { message: '请求参数验证失败 ', error: errorObj },
                HttpStatus.BAD_REQUEST,
            );
        }
        return value;
    }

    private toValidate(metatype: Type<any>): boolean {
        const types = [String, Boolean, Number, Array, Object];
        return !types.find(type => metatype === type);
    }
}
